(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Class types and default implementations for "mapper", "iterator", and
    "folder" over class definitions.

    Functions from [ClassDefinition] should be used for applications
    of "mapppers","iterators", and "folders". *)

class type class_definition_mapper = object
  method class_definition : AccessFlag.for_class list ->
    Name.for_class ->
    Name.for_class option ->
    Name.for_class list ->
    Field.t list ->
    Method.t list ->
    Attribute.for_class list ->
    (AccessFlag.for_class list * Name.for_class * Name.for_class option * Name.for_class list * Field.t list * Method.t list * Attribute.for_class list)
  method class_flags : AccessFlag.for_class list -> AccessFlag.for_class list
  method class_name : Name.for_class -> Name.for_class
  method class_extends : Name.for_class option -> Name.for_class option
  method class_implements : Name.for_class list -> Name.for_class list
  method class_fields : Field.t list -> Field.t list
  method class_field : Field.t -> Field.t
  method class_methods : Method.t list -> Method.t list
  method class_method : Method.t -> Method.t
  method class_attributes : Attribute.for_class list -> Attribute.for_class list
  method class_attribute : Attribute.for_class -> Attribute.for_class
  method field_flags : AccessFlag.for_field list -> AccessFlag.for_field list
  method field_name : Name.for_field -> Name.for_field
  method field_descriptor : Descriptor.for_field -> Descriptor.for_field
  method field_attributes : Attribute.for_field list -> Attribute.for_field list
  method field_attribute : Attribute.for_field -> Attribute.for_field
  method regular_method : Method.regular -> Method.regular
  method constructor_method : Method.constructor -> Method.constructor
  method initializer_method : Method.class_initializer -> Method.class_initializer
  method regular_method_flags : AccessFlag.for_method list -> AccessFlag.for_method list
  method regular_method_name : Name.for_method -> Name.for_method
  method regular_method_descriptor : Descriptor.for_method -> Descriptor.for_method
  method regular_method_attributes : Attribute.for_method list -> Attribute.for_method list
  method regular_method_attribute : Attribute.for_method -> Attribute.for_method
  method constructor_flags : AccessFlag.for_constructor list -> AccessFlag.for_constructor list
  method constructor_descriptor : Descriptor.for_parameter list -> Descriptor.for_parameter list
  method constructor_attributes : Attribute.for_method list -> Attribute.for_method list
  method constructor_attribute : Attribute.for_method -> Attribute.for_method
  method initializer_flags : AccessFlag.for_initializer list -> AccessFlag.for_initializer list
  method initializer_attributes : Attribute.for_method list -> Attribute.for_method list
  method initializer_attribute : Attribute.for_method -> Attribute.for_method
end
(** This class type defines a "mapper", instances being used as
    "functions" to transform class definitions. The "function" is defined
    {i by parts} through the various methods of the object. Any method is
    responsible for the calling of its embedded elements; this means that
    the method [class_definition] should call methods to map fields,
    attributes, etc. *)

class default_class_definition_mapper : class_definition_mapper
(** The default "mapper", that encodes the identity over class
    definitions. When inheriting from this class, one should not forget
    to call the parent methods in order to ensure that the whole
    structure is mapped. *)    

class type class_definition_iterator = object
  method class_definition : AccessFlag.for_class list ->
    Name.for_class ->
    Name.for_class option ->
    Name.for_class list ->
    Field.t list ->
    Method.t list ->
    Attribute.for_class list ->
    unit
  method class_flags : AccessFlag.for_class list -> unit
  method class_name : Name.for_class -> unit
  method class_extends : Name.for_class option -> unit
  method class_implements : Name.for_class list -> unit
  method class_fields : Field.t list -> unit
  method class_field : Field.t -> unit
  method class_methods : Method.t list -> unit
  method class_method : Method.t -> unit
  method class_attributes : Attribute.for_class list -> unit
  method class_attribute : Attribute.for_class -> unit
  method field_flags : AccessFlag.for_field list -> unit
  method field_name : Name.for_field -> unit
  method field_descriptor : Descriptor.for_field -> unit
  method field_attributes : Attribute.for_field list -> unit
  method field_attribute : Attribute.for_field -> unit
  method regular_method : Method.regular -> unit
  method constructor_method : Method.constructor -> unit
  method initializer_method : Method.class_initializer -> unit
  method regular_method_flags : AccessFlag.for_method list -> unit
  method regular_method_name : Name.for_method -> unit
  method regular_method_descriptor : Descriptor.for_method -> unit
  method regular_method_attributes : Attribute.for_method list -> unit
  method regular_method_attribute : Attribute.for_method -> unit
  method constructor_flags : AccessFlag.for_constructor list -> unit
  method constructor_descriptor : Descriptor.for_parameter list -> unit
  method constructor_attributes : Attribute.for_method list -> unit
  method constructor_attribute : Attribute.for_method -> unit
  method initializer_flags : AccessFlag.for_initializer list -> unit
  method initializer_attributes : Attribute.for_method list -> unit
  method initializer_attribute : Attribute.for_method -> unit
end
(** This class type defines an "iterator", instances being used as
    "functions" to iterate over the different components of a class
    definitions. The "function" is defined by parts through the various
    methods of the object. Any method is responsible for the calling of
    its embedded elements; this means that the method [class_definition]
    should call methods to iterate over fields, attributes, etc. *)

class default_class_definition_iterator : class_definition_iterator
(** The default "iterator", that iterates over the whole structure by
    doing nothing. When inheriting from this class, one should not forget
    to call the parent methods in order to ensure that the whole
    structure is traversed. *)

class type class_definition_folder = object ('self)
  method class_definition : AccessFlag.for_class list ->
    Name.for_class ->
    Name.for_class option ->
    Name.for_class list ->
    Field.t list ->
    Method.t list ->
    Attribute.for_class list ->
    'self
  method class_flags : AccessFlag.for_class list -> 'self
  method class_name : Name.for_class -> 'self
  method class_extends : Name.for_class option -> 'self
  method class_implements : Name.for_class list -> 'self
  method class_fields : Field.t list -> 'self
  method class_field : Field.t -> 'self
  method class_methods : Method.t list -> 'self
  method class_method : Method.t -> 'self
  method class_attributes : Attribute.for_class list -> 'self
  method class_attribute : Attribute.for_class -> 'self
  method field_flags : AccessFlag.for_field list -> 'self
  method field_name : Name.for_field -> 'self
  method field_descriptor : Descriptor.for_field -> 'self
  method field_attributes : Attribute.for_field list -> 'self
  method field_attribute : Attribute.for_field -> 'self
  method regular_method : Method.regular -> 'self
  method constructor_method : Method.constructor -> 'self
  method initializer_method : Method.class_initializer -> 'self
  method regular_method_flags : AccessFlag.for_method list -> 'self
  method regular_method_name : Name.for_method -> 'self
  method regular_method_descriptor : Descriptor.for_method -> 'self
  method regular_method_attributes : Attribute.for_method list -> 'self
  method regular_method_attribute : Attribute.for_method -> 'self
  method constructor_flags : AccessFlag.for_constructor list -> 'self
  method constructor_descriptor : Descriptor.for_parameter list -> 'self
  method constructor_attributes : Attribute.for_method list -> 'self
  method constructor_attribute : Attribute.for_method -> 'self
  method initializer_flags : AccessFlag.for_initializer list -> 'self
  method initializer_attributes : Attribute.for_method list -> 'self
  method initializer_attribute : Attribute.for_method -> 'self
end
(** This class type defines a "folder", instances being used as
    "functions" to fold over the different components of a class
    definitions. The "function" is defined by parts through the various
    methods of the object. Any method is responsible for the calling of
    its embedded elements; this means that the method [class_definition]
    should call methods to fold fields, attributes, etc. *)

class default_class_definition_folder : class_definition_folder
(** The default "folder", that folds over the whole structure by
    doing nothing. When inheriting from this class, one should not forget
    to call the parent methods in order to ensure that the whole
    structure is traversed. *)
