(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


let names = [ @"assemble"; @"asm" ]

let description = @"compiles to class files"

let latex_description = [
  @"Assembles (\\ie compiles) the passed assembler files into \\java{}" ;
  @"bytecode class files."
]

let versions =
  List.map
    (fun x -> Version.to_utf8 x, x)
    Version.all

class parameters elements_desc = object (self)
  inherit Command.base_parameters elements_desc as super

  val mutable target = Version.default

  val mutable compute_stacks = false

  val mutable optimize = false

  val mutable optimize_partial = false

  val mutable destination = @"."

  method set_target t =
    target <- t

  method get_target =
    target

  method set_compute_stacks () =
    compute_stacks <- true

  method is_compute_stacks =
    compute_stacks

  method set_optimize () =
    optimize <- true

  method is_optimize =
    optimize

  method set_optimize_partial () =
    optimize_partial <- true

  method is_optimize_partial =
    optimize_partial

  method set_destination d =
    destination <- d

  method get_destination =
    destination

  method! switches =
    let open UArg in
    [ { identifier = @"-compute-stacks";
        kind = Nothing self#set_compute_stacks;
        documentation = @"Computes stack elements (sizes and frames)"; } ;
      { identifier = @"-d";
        kind = String { switch_desc = @"<path>";
                        handle_string = self#set_destination; };
        documentation = @"Output path for generated class files"; } ;
      { identifier = @"-destination";
        kind = String { switch_desc = @"<path>";
                        handle_string = self#set_destination; };
        documentation = @"Output path for generated class files"; } ;
      { identifier = @"-optimize";
        kind = Nothing self#set_optimize;
        documentation = @"Optimize bytecode"; } ;
      { identifier = @"-optimize-partial";
        kind = Nothing self#set_optimize_partial;
        documentation = @"Optimize bytecode using partial evaluation"; } ;
      { identifier = @"-target";
        kind = Choice { switch_options = List.map fst versions;
                        handle_choice = (fun str ->
                          self#set_target (Utils.list_assoc UTF8.equal str versions)); };
        documentation = @"Target version for generated class files"; } ]
    @ super#switches

end

let make_parameters, switches_of_parameters, others_of_parameters =
  Command.parameters_functions
    (fun () ->
      new parameters
      @"arguments are paths of files to compile")

let run params =
  let source_files = List.map Path.make_of_utf8 params#get_elements in
  match source_files with
  | _ :: _ ->
      let class_path = params#build_class_path in
      let class_loader = ClassLoader.make_of_class_path class_path in
      let to_string class_name =
        try
          class_name
          |> Name.external_utf8_for_class
          |> UTF8.to_string
        with _ ->
          "<unrepresentable class name>" in
      List.iter
        (fun source_file ->
          let source_file = UTF8LineReader.make_of_path source_file in
          let class_name =
            Assembler.assemble
              ~version:params#get_target
              ~compute_stacks:params#is_compute_stacks
              ~optimize:params#is_optimize
              ~optimize_partial:params#is_optimize_partial
              ~class_loader
              source_file
              (Assembler.Path params#get_destination) in
          Printf.printf "%S has been compiled\n" (to_string class_name);
          UTF8LineReader.close_noerr source_file)
        source_files;
      ClassPath.close_noerr class_path
  | [] ->
      ()
