(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


let (+=) = Utils.(+=)
let (++) = UTF8.(++)

BARISTA_ERROR =
  | Duplicate_entry of (str : UTF8.t) ->
      Printf.sprintf "Duplicate entry %S"
        (UTF8.to_string_noerr str)
  | Duplicate_service of (str : UTF8.t) ->
      Printf.sprintf "Duplicate service %S"
        (UTF8.to_string_noerr str)

type t = {
    filename : UTF8.t;
    arch_stream : ArchiveOutputStream.t;
    merge_services : bool;
    fail_on_duplicate : bool;
    generate_index : bool;
    mutable entries : UTF8.Set.t;
   (* the following maps entry names to file contents
      (in reversed order) *)
    mutable services : Bytes.t list UTF8.Map.t;
  }

let service_prefix = @"META-INF/services/"

let index_name = @"META-INF/INDEX.LIST"

let make ?(merge_services = false) ?(fail_on_duplicate = false) ?(generate_index = false) path =
  let filename = Path.to_utf8 path in
  let arch_stream = ArchiveOutputStream.make_of_path path in
  { filename;
    arch_stream;
    merge_services;
    fail_on_duplicate;
    generate_index;
    entries = UTF8.Set.empty;
    services = UTF8.Map.empty }

let check_duplicate builder s =
  if builder.fail_on_duplicate && (UTF8.Set.mem s builder.entries) then
    fail (Duplicate_entry s);
  builder.entries <- UTF8.Set.add s builder.entries

(* returns [true] if NEIHTER a service NOR a manifest,
   adding data to service if name designates a service *)
let handle_service builder data name =
  let is_service = UTF8.starts_with service_prefix name in
  if is_service then begin
    let old_contents =
      try
        UTF8.Map.find name builder.services
      with Not_found ->
        [] in
    let new_contents = (Lazy.force data) :: old_contents in
    builder.services <- UTF8.Map.add name new_contents builder.services;
    false
  end else if UTF8.equal name Manifest.path_in_archive then
    false
  else
    true

let add_entry builder ?(prefix = @"") name bytes =
  let full_name = prefix ++ name in
  if handle_service builder (lazy bytes) full_name then begin
    check_duplicate builder full_name;
    ArchiveOutputStream.add_entry builder.arch_stream full_name bytes
  end

let add_entry_from_file builder ?(prefix = @"") name path =
  let full_name = prefix ++ name in
  if handle_service builder (lazy (OS.read_file path)) full_name then begin
    check_duplicate builder full_name;
    ArchiveOutputStream.add_entry_from_file builder.arch_stream full_name path
  end

let add_entry_from_class builder ?(prefix = @"") def =
  let buff = ByteBuffer.make_of_size 4096 in
  let stream = OutputStream.make_of_buffer buff in
  ClassFile.write (ClassDefinition.encode def) stream;
  OutputStream.flush stream;
  let bytes = ByteBuffer.contents buff in
  let name = Name.internal_utf8_for_class def.ClassDefinition.name in
  let full_name = prefix ++ name ++ @".class" in
  check_duplicate builder full_name;
  ArchiveOutputStream.add_entry builder.arch_stream full_name bytes

let add_entries_from_archive builder ?(prefix = @"") arch =
  ArchiveFile.iter_entries
    (fun entry ->
      if not (ArchiveEntry.is_directory entry) then
        let bytes = ArchiveFile.bytes_of_entry arch entry in
        let name = ArchiveEntry.get_filename entry in
        let full_name = prefix ++ name in
        if handle_service builder (lazy bytes) full_name then begin
          check_duplicate builder full_name;
          ArchiveOutputStream.add_entry builder.arch_stream full_name bytes
        end)
    arch

let add_entries_from_archive_file builder ?(prefix = @"") path =
  let arch = ArchiveFile.make_of_path path in
  add_entries_from_archive builder ~prefix arch;
  ArchiveFile.close_noerr arch

let add_entries_from_archive_files builder ?(prefix = @"") paths =
  List.iter
    (fun path -> add_entries_from_archive_file builder ~prefix path)
    paths

let add_manifest builder manifest =
  check_duplicate builder Manifest.path_in_archive;
  manifest
  |> Manifest.to_utf8
  |> UTF8.to_latin1
  |> ArchiveOutputStream.add_entry builder.arch_stream Manifest.path_in_archive

let close builder =
  UTF8.Map.iter
    (fun service_name service_contents ->
      if (not builder.merge_services) && ((List.length service_contents) > 1) then
        fail (Duplicate_service service_name)
      else begin
        let added = ref 0 in
        let data = ByteBuffer.make_of_size 4096 in
        List.iter
          (fun service_bytes ->
            if !added > 0 then
              ByteBuffer.add_byte data 10; (* new line *)
            ByteBuffer.add_bytes data service_bytes;
            added += Bytes.length service_bytes)
          (List.rev service_contents);
        let bytes = ByteBuffer.contents data in
        ArchiveOutputStream.add_entry builder.arch_stream service_name bytes
      end)
    builder.services;
  if builder.generate_index then begin
    let index_buff = UTF8Buffer.make_of_size 1024 in
    UTF8Buffer.add_string index_buff @"JarIndex-Version: 1.0\n\n";
    UTF8Buffer.add_endline index_buff builder.filename;
    UTF8.Set.iter
      (fun entry_name ->
        if not (UTF8.equal entry_name Manifest.path_in_archive) then
          UTF8Buffer.add_endline index_buff entry_name)
      builder.entries;
    UTF8Buffer.add_newline index_buff;
    index_buff
    |> UTF8Buffer.contents
    |> UTF8.to_latin1
    |> ArchiveOutputStream.add_entry builder.arch_stream index_name
  end;
  ArchiveOutputStream.close builder.arch_stream

let close_noerr builder =
  try
    close builder
  with _ ->
    ()
