(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

IFNDEF USE_JDK THEN

let argv = Array.map UTF8.of_string Sys.argv

let get_env str =
  str
  |> UTF8.to_string
  |> Sys.getenv
  |> UTF8.of_string

let get_property _ =
  raise Not_found

let timestamp () =
  let now = Unix.localtime (Unix.gettimeofday ()) in
  Printf.sprintf "%04d-%02d-%02d"
    (now.Unix.tm_year + 1900)
    (succ now.Unix.tm_mon)
    (now.Unix.tm_mday)

let execute cmd =
  let temp = Filename.temp_file "barista" "javahome" in
  let code = Sys.command (cmd ^ " 2> /dev/null > " ^ temp) in
  if code <> 0 then raise Not_found;
  try
    let reader =
      temp
      |> Path.make_of_string
      |> UTF8LineReader.make_of_path in
    let res = UTF8LineReader.read_lines reader in
    UTF8LineReader.close_noerr reader;
    res
  with _ ->
    raise Not_found

ELSE (* USE_JDK *)

external java_string_of_utf8 : UTF8.t -> java'lang'String java_instance =
  "%identity"

external utf8_of_java_string : java'lang'String java_instance -> UTF8.t =
  "%identity"

external ocamljava_get_argv : unit ->  UTF8.t array =
  "ocamljava_get_argv"

let argv = ocamljava_get_argv ()

let get_env str =
  let res = Java.call "System.getenv(_)" (java_string_of_utf8 str) in
  if Java.is_null res then
    raise Not_found
  else
    utf8_of_java_string res

let get_property str =
  let res = Java.call "System.getProperty(_)" (java_string_of_utf8 str) in
  if Java.is_null res then
    raise Not_found
  else
    utf8_of_java_string res

let timestamp () =
  let calendar = Java.call "java.util.Calendar.getInstance()" () in
  let year =
    Java.call "java.util.Calendar.get(_)"
      calendar
      (Java.get "java.util.Calendar.YEAR" ()) in
  let month =
    Java.call "java.util.Calendar.get(_)"
      calendar
      (Java.get "java.util.Calendar.MONTH" ()) in
  let day =
    Java.call "java.util.Calendar.get(_)"
      calendar
      (Java.get "java.util.Calendar.DAY_OF_MONTH" ()) in
  Printf.sprintf "%04ld-%02ld-%02ld"
    year
    (Int32.succ month)
    day

external javastring_of_string : string -> _'String java_instance =
  "ocamljava_javastring_of_string"

external utf8_of_java_string : java'lang'String java_instance -> UTF8.t =
  "%identity"

let execute cmd =
  let args = Java.make_array "String[]" 1l in
  JavaReferenceArray.set args 0l (javastring_of_string cmd);
  let reader =
    args
    |> Java.make "ProcessBuilder(String[])"
    |> Java.call "ProcessBuilder.start()"
    |> Java.call "Process.getInputStream()"
    |> Java.make "java.io.InputStreamReader(_)"
    |> Java.make "java.io.BufferedReader(_)" in
  let rec read acc =
    let line = Java.call "java.io.BufferedReader.readLine()" reader in
    if Java.is_null line then
      List.rev acc
    else
      read ((utf8_of_java_string line) :: acc) in
  read []

END

let read_file path =
  let res = ByteBuffer.make_of_size 4096 in
  let buff = Bytes.make_of_size 1024 in
  let stream = InputStream.make_of_path path in
  let read = ref (-1) in
  while !read <> 0 do
    read := InputStream.read_available_bytes stream 1024 buff 0;
    ByteBuffer.add_bytes_from res buff 0 !read
  done;
  ByteBuffer.contents res
