(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** File paths.

    This definition is independent of the actual implementation, that can
    be based on either the OCaml standard library, or the OCaml-Java
    runtime library. *)


type t
(** The type of file paths. *)

val make_of_string : string -> t
(** Creates a path from a string. *)

val make_of_utf8 : UTF8.t -> t
(** Creates a path from a UTF8 string. *)

val basename : t -> UTF8.t
(** Returns the base name of the passed path. *)

val dirname : t -> t
(** Returns the parent directory of the passed path. *)

val absolute : t -> t
(** Returns a path equivalent to the passed one, except that it is
    absolute. *)

val is_directory : t -> bool
(** Tests whether the passed path is a directory. *)

val is_file : t -> bool
(** Tests whether the passed path is a file. *)

val exists : t -> bool
(** Tests whether the passed path designates an existing file or
    directory. *)

val check_suffix : UTF8.t -> t -> bool
(** [check_suffix suffix path] returns [true] iff the passed path ends
    with [suffix]. *)

val concat : t -> t -> t
(** Returns the concatenation of the passed paths. *)

val mkdirs : t -> unit
(** Creates all the needed directories to make the passed path actually
    exist, as a directory.

    Raises [Sys_error] if a directory cannot be created. *)

val to_string : t -> string
(** Converts a path into a string. *)

val to_utf8 : t -> UTF8.t
(** Converts a path into a UTF8 string. *)

val equal : t -> t -> bool
(* Equality over paths. *)

val compare : t -> t -> int
(* Comparison over paths. *)

val hash : t -> int
(* Hash function over paths. *)
