(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

IFNDEF USE_JDK THEN

type t = CamomileLibrary.UTF8.Buf.buf

external camomile_of_uchar : UChar.t -> CamomileLibrary.UChar.t =
  "%identity"

external camomile_of_utf8 : UTF8.t -> CamomileLibrary.UTF8.t =
  "%identity"

external utf8_of_camomile : CamomileLibrary.UTF8.t -> UTF8.t =
  "%identity"

let default_size = 1024

let make () =
  CamomileLibrary.UTF8.Buf.create default_size

let make_of_size sz =
  if sz < 0 then invalid_arg "BaristaLibrary.UTF8Buffer.make_of_size";
  CamomileLibrary.UTF8.Buf.create sz

let add_char buff ch =
  ch
  |> camomile_of_uchar
  |> CamomileLibrary.UTF8.Buf.add_char buff

let add_string buff str =
  str
  |> camomile_of_utf8
  |> CamomileLibrary.UTF8.Buf.add_string buff

let eol = UChar.of_char '\n'

let add_endline buff str =
  add_string buff str;
  add_char buff eol

let add_newline buff =
  add_char buff eol

let contents buff =
  buff
  |> CamomileLibrary.UTF8.Buf.contents
  |> utf8_of_camomile

let add_string2 buff str =
  add_string buff str; buff

let add_padding buff n pad_char =
  for _i = 1 to n do
    add_char buff pad_char
  done;
  buff

let print buff left pad pad_char str =
  let str = UTF8.of_string str in
  let len = UTF8.length str in
  let pad = pad - len in
  if left then begin
    add_string buff str;
    add_padding buff pad pad_char
  end else begin
    ignore (add_padding buff pad pad_char);
    add_string2 buff str
  end

let print_string buff escape left pad str =
  let str = if escape then UTF8.escape str else str in
  let len = UTF8.length str in
  let pad = pad - len in
  if left then begin
    add_string buff str;
    add_padding buff pad @' '
  end else begin
    ignore (add_padding buff pad @' ');
    add_string2 buff str
  end

let add_char2 buff ch =
  add_char buff ch; buff

let print_char buff escape left pad ch =
  print_string buff escape left pad (UTF8.of_uchar ch)

let string_of_float f =
  let res = string_of_float f in
  let len = String.length res in
  if (len > 0) && (res.[pred len] = '.') then
    res ^ "0"
  else
    res

let add_float buff f =
  f
  |> string_of_float
  |> UTF8.of_string
  |> add_string2 buff

let print_float buff left pad f =
  print buff left pad @' ' (string_of_float f)

let add_int buff x =
  x
  |> string_of_int
  |> UTF8.of_string
  |> add_string2 buff

let add_int_hexa buff x =
  x
  |> Printf.sprintf "%xd"
  |> UTF8.of_string
  |> add_string2 buff

let print_int buff left pad zeroes hexa x =
  print buff left pad
    (if zeroes then @'0' else @' ')
    (if hexa then
      Printf.sprintf "%xd" x
    else
      string_of_int x)

let add_int32 buff x =
  x
  |> Int32.to_string
  |> UTF8.of_string
  |> add_string2 buff

let add_int32_hexa buff x =
  x
  |> Printf.sprintf "%lxd"
  |> UTF8.of_string
  |> add_string2 buff

let print_int32 buff left pad zeroes hexa x =
  print buff left pad
    (if zeroes then @'0' else @' ')
    (if hexa then
      Printf.sprintf "%lxd" x
    else
      Int32.to_string x)

let add_int64 buff x =
  x
  |> Int64.to_string
  |> UTF8.of_string
  |> add_string2 buff

let add_int64_hexa buff x =
  x
  |> Printf.sprintf "%Lxd"
  |> UTF8.of_string
  |> add_string2 buff

let print_int64 buff left pad zeroes hexa x =
  print buff left pad
    (if zeroes then @'0' else @' ')
    (if hexa then
      Printf.sprintf "%Lxd" x
    else
      Int64.to_string x)

let add_nativeint buff x =
  x
  |> Nativeint.to_string
  |> UTF8.of_string
  |> add_string2 buff

let add_nativeint_hexa buff x =
  x
  |> Printf.sprintf "%nxd"
  |> UTF8.of_string
  |> add_string2 buff

let print_nativeint buff left pad zeroes hexa x =
  print buff left pad
    (if zeroes then @'0' else @' ')
    (if hexa then
      Printf.sprintf "%nxd" x
    else
      Nativeint.to_string x)

let add_newline2 buff =
  add_newline buff; buff

ELSE (* USE_JDK *)

type t = java'lang'StringBuilder java_instance

external code_point_of_uchar : UChar.t -> int32 =
  "%identity"

external java_string_of_utf8 : UTF8.t -> java'lang'Object java_extends =
  "%identity"

external utf8_of_java_string : java'lang'Object java_extends -> UTF8.t =
  "%identity"

let default_size = 1024

let make () =
  default_size
  |> Int32.of_int
  |> Java.make "StringBuilder(int)"

let make_of_size sz =
  if sz < 0 then invalid_arg "BaristaLibrary.UTF8Buffer.make_of_size";
  sz
  |> Int32.of_int
  |> Java.make "StringBuilder(int)"

let add_char buff ch =
  ch
  |> code_point_of_uchar
  |> Java.call "StringBuilder.appendCodePoint(_):StringBuilder" buff
  |> ignore

let add_string buff str =
  str
  |> java_string_of_utf8
  |> Java.call "StringBuilder.append(String):StringBuilder" buff
  |> ignore

let add_endline buff str =
  str
  |> java_string_of_utf8
  |> Java.call "StringBuilder.append(String):StringBuilder" buff
  |> ignore;
  Java.call "StringBuilder.appendCodePoint(_):StringBuilder" buff 10l
  |> ignore

let add_newline buff =
  Java.call "StringBuilder.appendCodePoint(_):StringBuilder" buff 10l
  |> ignore

let contents buff =
  buff
  |> Java.call "StringBuilder.toString()"
  |> utf8_of_java_string

let add_string2 buff str =
  str
  |> java_string_of_utf8
  |> Java.call "StringBuilder.append(String):StringBuilder" buff

let add_padding buff n pad_char =
  for _i = 1 to n do
    Java.call "StringBuilder.appendCodePoint(_):StringBuilder" buff pad_char
    |> ignore
  done;
  buff

let print buff left pad pad_char str =
  let len = Int32.to_int (Java.call "String.length()" str) in
  let pad = pad - len in
  if left then begin
    Java.call "StringBuilder.append(String):StringBuilder" buff str
    |> ignore;
    add_padding buff pad pad_char
  end else begin
    add_padding buff pad pad_char
    |> ignore;
    Java.call "StringBuilder.append(String):StringBuilder" buff str
  end

let print_string buff escape left pad str =
  let str = if escape then UTF8.escape str else str in
  let len = UTF8.length str in
  let pad = pad - len in
  if left then begin
    add_string buff str;
    add_padding buff pad 32l
  end else begin
    ignore (add_padding buff pad 32l);
    add_string2 buff str
  end

let add_char2 buff ch =
  ch
  |> code_point_of_uchar
  |> Java.call "StringBuilder.appendCodePoint(_):StringBuilder" buff

let print_char buff escape left pad ch =
  print_string buff escape left pad (UTF8.of_uchar ch)

let add_float buff f =
  f
  |> Java.call "StringBuilder.append(double):StringBuilder" buff

let print_float buff left pad f =
  print buff left pad 32l
    (Java.call "Float.toString(_)" f)

let add_int buff x =
  x
  |> Int64.of_int
  |> Java.call "StringBuilder.append(long):StringBuilder" buff

let add_int_hexa buff x =
  Java.call "Long.toString(_,_)" (Int64.of_int x) 16l
  |> Java.call "StringBuilder.append(String):StringBuilder" buff

let print_int buff left pad zeroes hexa x =
  print buff left pad
    (if zeroes then 48l else 32l)
    (Java.call "Long.toString(_,_)"
       (Int64.of_int x)
       (if hexa then 16l else 10l))

let add_int32 buff x =
  x
  |> Java.call "StringBuilder.append(int):StringBuilder" buff

let add_int32_hexa buff x =
  Java.call "Integer.toString(_,_)" x 16l
  |> Java.call "StringBuilder.append(String):StringBuilder" buff

let print_int32 buff left pad zeroes hexa x =
  print buff left pad
    (if zeroes then 48l else 32l)
    (Java.call "Integer.toString(_,_)" x (if hexa then 16l else 10l))

let add_int64 buff x =
  x
  |> Java.call "StringBuilder.append(long):StringBuilder" buff

let add_int64_hexa buff x =
  Java.call "Long.toString(_,_)" x 16l
  |> Java.call "StringBuilder.append(String):StringBuilder" buff

let print_int64 buff left pad zeroes hexa x =
  print buff left pad
    (if zeroes then 48l else 32l)
    (Java.call "Long.toString(_,_)" x (if hexa then 16l else 10l))

let add_nativeint buff x =
  x
  |> Int64.of_nativeint
  |> Java.call "StringBuilder.append(long):StringBuilder" buff

let add_nativeint_hexa buff x =
  Java.call "Long.toString(_,_)" (Int64.of_nativeint x) 16l
  |> Java.call "StringBuilder.append(String):StringBuilder" buff

let print_nativeint buff left pad zeroes hexa x =
  print buff left pad
    (if zeroes then 48l else 32l)
    (Java.call "Long.toString(_,_)"
       (Int64.of_nativeint x)
       (if hexa then 16l else 10l))

let add_newline2 buff =
  Java.call "StringBuilder.appendCodePoint(_):StringBuilder" buff 10l

END
