(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

IFNDEF USE_JDK THEN

BARISTA_ERROR =
  | Unable_to_open_reader of (path : Path.t) ->
      Printf.sprintf "unable to open reader (%S)"
        (Path.to_string path)
  | Unable_to_read_data ->
      "unable to read data"
  | Unable_to_close_reader ->
      "unable to close reader"

module Encoding =
  CamomileLibrary.CharEncoding.Configure (CamomileLibraryDefault.Config)

module UTF8Line = CamomileLibrary.ULine.Make (CamomileLibrary.UTF8)

type t = UTF8Line.input_line

external bytes_of_string : string -> Bytes.t =
  "%identity"

external string_of_path : Path.t -> string =
  "%identity"

class input_channel chan = object

    method input str ofs len =
      let bytes = bytes_of_string str in
      let res = InputStream.read_available_bytes chan len bytes ofs in
      if res > 0 then
        res
      else
        raise End_of_file

    method close_in () =
      InputStream.close chan

end

let make_of_stream stream =
  stream
  |> new input_channel
  |> new Encoding.uchar_input_channel_of Encoding.utf8
  |> new UTF8Line.input_line

let make_of_path path =
  (try
    path
    |> string_of_path
    |> open_in
  with _ ->
    fail (Unable_to_open_reader path))
  |> InputStream.make_of_channel
  |> make_of_stream

external utf8_of_camomile : CamomileLibrary.UTF8.t -> UTF8.t =
  "%identity"

let read_line reader =
  try
    reader#get ()
    |> utf8_of_camomile
  with
  | End_of_file as eof -> raise eof
  | _ -> fail Unable_to_read_data

let read_lines reader =
  let res = ref [] in
  try
    while true do
      res := (utf8_of_camomile (reader#get ())) :: !res
    done;
    assert false
  with End_of_file ->
    List.rev !res

let close reader =
  try
    reader#close_in ()
  with _ ->
    fail Unable_to_close_reader

let close_noerr reader =
  try
    reader#close_in ()
  with _ ->
    ()

ELSE (* USE_JDK *)

BARISTA_ERROR =
  | Unable_to_open_reader of (path : Path.t) ->
      Printf.sprintf "unable to open reader (%S)"
        (Path.to_string path)
  | Unable_to_read_data ->
      "unable to read data"
  | Unable_to_close_reader ->
      "unable to close reader"

type t = java'io'BufferedReader java_instance

external utf8_of_java_string : java'lang'String java_instance -> UTF8.t =
  "%identity"

external data_input_stream_of_input_stream : InputStream.t -> java'io'InputStream java_instance =
  "%identity"

external file_of_path : Path.t -> java'io'File java_instance =
  "%identity"

let make_of_path path =
  let file =
    path
    |> file_of_path in
  let files_state =
    Java.call "org.ocamljava.runtime.context.CurrentContext.getFilesState()" () in
  let hook =
    Java.call "org.ocamljava.runtime.context.FilesState.getFileHook()" files_state in
  let stream =
    if Java.is_not_null hook then
      Some (file
           |> Java.call "org.ocamljava.runtime.context.FilesState.resourceNameFromFile(_)" files_state
           |> Java.call "org.ocamljava.runtime.context.FileHook.getInputStream(_)" hook)
    else
      None in
  match stream with
  | Some st when Java.is_not_null st ->
      st
      |> Java.make "java.io.InputStreamReader(_)"
      |> Java.make "java.io.BufferedReader(_)"
  | Some _ | None ->
      try
        file
        |> Java.make "java.io.FileReader(java.io.File)"
        |> Java.make "java.io.BufferedReader(_)"
      with Java_exception _ ->
        fail (Unable_to_open_reader path)

let make_of_stream stream =
  stream
  |> data_input_stream_of_input_stream
  |> Java.make "java.io.InputStreamReader(_)"
  |> Java.make "java.io.BufferedReader(_)"

let read_line reader =
  let res =
    try
      Java.call "java.io.BufferedReader.readLine()" reader
    with Java_exception _ ->
      fail Unable_to_read_data in
  if Java.is_null res then
    raise End_of_file
  else
    utf8_of_java_string res

let read_lines reader =
  try
    let res = ref [] in
    let last = ref (Java.call "java.io.BufferedReader.readLine()" reader) in
    while Java.is_not_null !last do
      res := (utf8_of_java_string !last) :: !res;
      last := Java.call "java.io.BufferedReader.readLine()" reader
    done;
    List.rev !res
  with Java_exception _ ->
    fail Unable_to_read_data

let close reader =
  try
    Java.call "java.io.BufferedReader.close()" reader
  with Java_exception _ ->
    fail Unable_to_close_reader

let close_noerr reader =
  try
    Java.call "java.io.BufferedReader.close()" reader
  with Java_exception _ ->
    ()

END
