(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

type archive_entry =
  | Directory_entry of ArchiveEntry.t
  | Package_entry of ArchiveEntry.t * Bytes.t
  | Class_entry of ArchiveEntry.t * Bytes.t
  | Bare_entry of ArchiveEntry.t * Bytes.t

class type archive_mapper = object
  method comment : UTF8.t -> UTF8.t
  method entry : ArchiveEntry.t -> archive_entry
  method directory_entry : ArchiveEntry.t -> ArchiveEntry.t
  method package_entry : ArchiveEntry.t -> ArchiveEntry.t
  method class_entry : ArchiveEntry.t -> ArchiveEntry.t
  method bare_entry : ArchiveEntry.t -> ArchiveEntry.t
  method package_definition : PackageDefinition.t -> PackageDefinition.t
  method class_definition : ClassDefinition.t -> ClassDefinition.t
  method data : Bytes.t -> Bytes.t
end

let default_buffer_size = 65536

class default_archive_mapper z = object (self)

  method comment (x : UTF8.t) = x

  method entry x =
    let filename = ArchiveEntry.get_filename x in
    if ArchiveEntry.is_directory x then begin
      Directory_entry (self#directory_entry x)
    end else if UTF8.ends_with @"package-info.class" filename then begin
      let e = self#package_entry x in
      let is = ArchiveFile.stream_of_entry z x in
      let cf = ClassFile.read is in
      let pd = PackageDefinition.decode cf in
      let pd = self#package_definition pd in
      let cf = PackageDefinition.encode pd in
      let b = ByteBuffer.make_of_size default_buffer_size in
      let os = OutputStream.make_of_buffer b in
      ClassFile.write cf os;
      OutputStream.flush os;
      Package_entry (e, ByteBuffer.contents b)
    end else if UTF8.ends_with @".class" filename then begin
      let e = self#class_entry x in
      let is = ArchiveFile.stream_of_entry z x in
      let cf = ClassFile.read is in
      let cd = ClassDefinition.decode cf in
      let cd = self#class_definition cd in
      let cf = ClassDefinition.encode cd in
      let b = ByteBuffer.make_of_size default_buffer_size in
      let os = OutputStream.make_of_buffer b in
      ClassFile.write cf os;
      OutputStream.flush os;
      Class_entry (e, ByteBuffer.contents b)
    end else begin
      let e = self#bare_entry x in
      let d = ArchiveFile.bytes_of_entry z x in
      let d = self#data d in
      Bare_entry (e, d)
    end

  method directory_entry x = x

  method package_entry x = x

  method class_entry x = x

  method bare_entry x = x

  method package_definition x = x

  method class_definition x = x

  method data x = x

end

class type archive_iterator = object
  method comment : UTF8.t -> unit
  method entry : ArchiveEntry.t -> unit
  method directory_entry : ArchiveEntry.t -> unit
  method package_entry : ArchiveEntry.t -> unit
  method class_entry : ArchiveEntry.t -> unit
  method bare_entry : ArchiveEntry.t -> unit
  method package_definition : PackageDefinition.t -> unit
  method class_definition : ClassDefinition.t -> unit
  method data : Bytes.t -> unit
end

class default_archive_iterator z = object (self)

  method comment (_ : UTF8.t) = ()

  method entry x =
    let filename = ArchiveEntry.get_filename x in
    if ArchiveEntry.is_directory x then begin
      self#directory_entry x
    end else if UTF8.ends_with @"package-info.class" filename then begin
      self#package_entry x;
      let is = ArchiveFile.stream_of_entry z x in
      let cf = ClassFile.read is in
      let pd = PackageDefinition.decode cf in
      self#package_definition pd
    end else if UTF8.ends_with @".class" filename then begin
      self#class_entry x;
      let is = ArchiveFile.stream_of_entry z x in
      let cf = ClassFile.read is in
      let cd = ClassDefinition.decode cf in
      self#class_definition cd
    end else begin
      self#bare_entry x;
      let d = ArchiveFile.bytes_of_entry z x in
      self#data d
    end

  method directory_entry _ = ()

  method package_entry _ = ()

  method class_entry _ = ()

  method bare_entry _ = ()

  method package_definition _ = ()

  method class_definition _ = ()

  method data _ = ()

end

class type archive_folder = object ('self)
  method comment : UTF8.t -> 'self
  method entry : ArchiveEntry.t -> 'self
  method directory_entry : ArchiveEntry.t -> 'self
  method package_entry : ArchiveEntry.t -> 'self
  method class_entry : ArchiveEntry.t -> 'self
  method bare_entry : ArchiveEntry.t -> 'self
  method package_definition : PackageDefinition.t -> 'self
  method class_definition : ClassDefinition.t -> 'self
  method data : Bytes.t -> 'self
end

class default_archive_folder z = object (self)

  method comment (_ : UTF8.t) = self

  method entry x =
    let filename = ArchiveEntry.get_filename x in
    if ArchiveEntry.is_directory x then begin
      self#directory_entry x
    end else if UTF8.ends_with @"package-info.class" filename then begin
      let self = self#package_entry x in
      let is = ArchiveFile.stream_of_entry z x in
      let cf = ClassFile.read is in
      let pd = PackageDefinition.decode cf in
      let self = self#package_definition pd in
      self
    end else if UTF8.ends_with @".class" filename then begin
      let self = self#class_entry x in
      let is = ArchiveFile.stream_of_entry z x in
      let cf = ClassFile.read is in
      let cd = ClassDefinition.decode cf in
      let self = self#class_definition cd in
      self
    end else begin
      let self = self#bare_entry x in
      let d = ArchiveFile.bytes_of_entry z x in
      let self = self#data d in
      self
    end

  method directory_entry _ = self

  method package_entry _ = self

  method class_entry _ = self

  method bare_entry _ = self

  method package_definition _ = self

  method class_definition _ = self

  method data _ = self

end

let map (m : archive_mapper) i o =
  let comment = m#comment (ArchiveFile.get_comment i) in
  let o = ArchiveOutputStream.make_of_path ~comment o in
  ArchiveFile.iter_entries
    (fun entry ->
      match m#entry entry with
      | Directory_entry _ -> ()
      | Package_entry (e, data)
      | Class_entry (e, data)
      | Bare_entry (e, data) ->
          ArchiveOutputStream.add_entry
            o
            ~extra:(ArchiveEntry.get_extra e)
            ~comment:(ArchiveEntry.get_comment e)
            (ArchiveEntry.get_filename e)
            data)
    i;
  o

let iter f i =
  f#comment (ArchiveFile.get_comment i);
  ArchiveFile.iter_entries f#entry i

let fold f i =
  let f = f#comment (ArchiveFile.get_comment i) in
  ArchiveFile.fold_entries
    (fun acc entry ->
      acc#entry entry)
    f
    i

let map_file mk ?(post = ignore) i o =
  let i = ArchiveFile.make_of_path i in
  let f = mk i in
  try
    let o = map f i o in
    post o;
    ArchiveFile.close_noerr i;
    ArchiveOutputStream.close_noerr o
  with e ->
    ArchiveFile.close_noerr i;
    raise e

let iter_file mk i =
  let i = ArchiveFile.make_of_path i in
  let f = mk i in
  try
    iter f i;
    ArchiveFile.close_noerr i
  with e ->
    ArchiveFile.close_noerr i;
    raise e

let fold_file mk i =
  let i = ArchiveFile.make_of_path i in
  let f = mk i in
  try
    let res = fold f i in
    ArchiveFile.close_noerr i;
    res
  with e ->
    ArchiveFile.close_noerr i;
    raise e
