(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Signatures definition as well as conversion functions from and to strings. *)


(** {6 Signatures definition} *)

(** The following types are direct mappings from the class file
    specification. One should refer to this document for the semantics
    of these types. *)

type class_signature = {
    formal_type_parameters : formal_type_parameter list;
    super_class_signature : class_type_signature;
    super_interface_signatures : class_type_signature list;
  }
and formal_type_parameter = {
    identifier : UTF8.t;
    class_bound : field_type_signature;
    interface_bounds : field_type_signature list;
  }
and field_type_signature =
  | Class_type_signature of class_type_signature
  | Array_type_signature of array_type_signature
  | Type_variable_signature of type_variable_signature
and class_type_signature = {
    qualified_class_name : Name.for_class;
    type_arguments : type_argument list;
    signature_suffix : class_type_signature_suffix list;
  }
and array_type_signature = type_signature
and type_signature =
  | Field_type_signature of field_type_signature
  | Base_type of Descriptor.java_type
and type_variable_signature = UTF8.t
and type_argument =
  | Star
  | Plus of field_type_signature
  | Minus of field_type_signature
  | Simple of field_type_signature
and class_type_signature_suffix = {
    suffix_identifier : UTF8.t;
    suffix_type_arguments : type_argument list;
  }
and method_signature = {
    formal_type_params : formal_type_parameter list;
    types : type_signature list;
    return : type_signature;
    throws_signatures : throws_signature list;
  }
and throws_signature =
  | Throws_class_type_signature of class_type_signature
  | Throws_type_variable_signature of type_variable_signature

val equal_class_signature : class_signature -> class_signature -> bool
(** Equality over class signatures. *)

val equal_formal_type_parameter : formal_type_parameter -> formal_type_parameter -> bool
(** Equality over formal type parameters. *)

val equal_field_type_signature : field_type_signature -> field_type_signature -> bool
(** Equality over field type signatures. *)

val equal_class_type_signature : class_type_signature -> class_type_signature -> bool
(** Equality over class type signatures. *)

val equal_array_type_signature : array_type_signature -> array_type_signature -> bool
(** Equality over array type signatures. *)

val equal_type_signature : type_signature -> type_signature -> bool
(** Equality over type signatures. *)

val equal_type_variable_signature : type_variable_signature -> type_variable_signature -> bool
(** Equality over type variable signatures. *)

val equal_type_argument : type_argument -> type_argument -> bool
(** Equality over type arguments. *)

val equal_class_type_signature_suffix : class_type_signature_suffix -> class_type_signature_suffix -> bool
(** Equality over class type signature suffixes. *)

val equal_method_signature : method_signature -> method_signature -> bool
(** Equality over method signatures. *)

val equal_throws_signature : throws_signature -> throws_signature -> bool
(** Equality over throws signatures. *)

val compare_class_signature : class_signature -> class_signature -> int
(** Comparison over class signatures. *)

val compare_formal_type_parameter : formal_type_parameter -> formal_type_parameter -> int
(** Comparison over formal type parameters. *)

val compare_field_type_signature : field_type_signature -> field_type_signature -> int
(** Comparison over field type signatures. *)

val compare_class_type_signature : class_type_signature -> class_type_signature -> int
(** Comparison over class type signatures. *)

val compare_array_type_signature : array_type_signature -> array_type_signature -> int
(** Comparison over array type signatures. *)

val compare_type_signature : type_signature -> type_signature -> int
(** Comparison over type signatures. *)

val compare_type_variable_signature : type_variable_signature -> type_variable_signature -> int
(** Comparison over type variable signatures. *)

val compare_type_argument : type_argument -> type_argument -> int
(** Comparison over type arguments. *)

val compare_class_type_signature_suffix : class_type_signature_suffix -> class_type_signature_suffix -> int
(** Comparison over class type signature suffixes. *)

val compare_method_signature : method_signature -> method_signature -> int
(** Comparison over method signatures. *)

val compare_throws_signature : throws_signature -> throws_signature -> int
(** Comparison over throws signature. *)

val hash_class_signature : class_signature -> int
(** Hash function over class signatures. *)

val hash_formal_type_parameter : formal_type_parameter -> int
(** Hash function over formal type parameters. *)

val hash_field_type_signature : field_type_signature -> int
(** Hash function over field type signatures. *)

val hash_class_type_signature : class_type_signature -> int
(** Hash function over class type signatures. *)

val hash_array_type_signature : array_type_signature -> int
(** Hash function over array type signatures. *)

val hash_type_signature : type_signature -> int
(** Hash function over type signatures. *)

val hash_type_variable_signature : type_variable_signature -> int
(** Hash function over type variable signatures. *)

val hash_type_argument : type_argument -> int
(** Hash function over type arguments. *)

val hash_class_type_signature_suffix : class_type_signature_suffix -> int
(** Hash function over class type signature suffixes. *)

val hash_method_signature : method_signature -> int
(** Hash function over method signatures. *)

val hash_throws_signature : throws_signature -> int
(** Hash function over throws signatures. *)


(** {6 Exception} *)

BARISTA_ERROR =
  | Invalid_signature_type_header of UChar.t
  | Invalid_signature_primitive_character of UChar.t
  | Invalid_signature_type of Descriptor.java_type
  | Invalid_class_signature of UTF8.t
  | Invalid_field_signature of UTF8.t
  | Invalid_method_signature of UTF8.t
  | Extra_elements_after_class_signature of UTF8.t
  | Extra_elements_after_field_signature of UTF8.t
  | Extra_elements_after_method_signature of UTF8.t


(** {6 Conversion functions} *)

val class_signature_of_utf8 : UTF8.t -> class_signature
(** Converts a string (as used in class file) into the corresponding signature.
    Raises [Exception] if conversion fails. *)

val utf8_of_class_signature : class_signature -> UTF8.t
(** Converts a signature into the corresponding string (as used in class file).
    Raises [Exception] if conversion fails. *)

val field_type_signature_of_utf8 : UTF8.t -> field_type_signature
(** Converts a string (as used in class file) into the corresponding signature.
    Raises [Exception] if conversion fails. *)

val utf8_of_field_type_signature : field_type_signature -> UTF8.t
(** Converts a signature into the corresponding string (as used in class file).
    Raises [Exception] if conversion fails. *)

val method_signature_of_utf8 : UTF8.t -> method_signature
(** Converts a string (as used in class file) into the corresponding signature.
    Raises [Exception] if conversion fails. *)

val utf8_of_method_signature : method_signature -> UTF8.t
(** Converts a signature into the corresponding string (as used in class file).
    Raises [Exception] if conversion fails. *)
