(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** XML documents, and dump to UTF8. *)

type pcdata = private UTF8.t
(** The type of bare text in an XML document. *)
and tag = private {
    tag : UTF8.t; (** Identifier of XML tag. *)
    attrs : (UTF8.t * UTF8.t) list; (** Attributes of XML tag. *)
    children : t list; (** Children of XML tag. *)
  }
(** The type of XML tags. *)
and t =
  | Pcdata of pcdata (** Bare text data. *)
  | Tag of tag (** Tag, with nested nodes. *)
(** The type of XML nodes. *)

val pcdata : UTF8.t -> t
(** [pcdata x] is equivalent to [Pcdata x]. *)

val tag : tag:UTF8.t -> attrs:(UTF8.t * UTF8.t) list -> children:t list -> t
(** [tag ~tag:t ~attrs:a ~children:c] is equivalent to
    [Tag { tag = t; attrs = a; children = c }]. *)

val escape : UTF8.t -> UTF8.t
(** Escapes the passed string in such a way it can be used as an
    attribute value. *)

val dump : t -> UTF8Buffer.t -> unit
(** Dumps the passed XML document to the passed buffer. *)

val to_utf8 : t -> UTF8.t
(** Converts the passed XML document into a UTF8 string. *)
