(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


let (++) = UTF8.(++)

(* Exception *)

BARISTA_ERROR =
  | Unable_to_open_archive of (path : Path.t) ->
      Printf.sprintf "unable to open archive %S"
        (UTF8.to_string_noerr (Path.to_utf8 path))
  | Does_not_exist of (path : Path.t) ->
      Printf.sprintf "%S does not exist"
        (UTF8.to_string_noerr (Path.to_utf8 path))
  | Class_not_found of (str : UTF8.t) ->
      Printf.sprintf "class %S not found"
        (UTF8.to_string_noerr str)


(* Types *)

type big_archive = {
    big_archive_prefix : UTF8.t;
    big_archive_file : ArchiveFile.t;
    big_archive_prefix_list : UTF8.t list lazy_t;
  }

type entry_source =
  | Core_library
  | System_classpath
  | Explicit

type 'a generic_entry = {
    entry_source : entry_source;
    entry_path : Path.t;
    entry_value : 'a;
  }

type entry =
  | Directory of unit generic_entry
  | Archive of ArchiveFile.t generic_entry
  | Big_archive of big_archive generic_entry
  | Error of error generic_entry

type t = entry list


(* Constructors *)

let system_classpath =
  try
    OS.get_env @"CLASSPATH"
  with Not_found ->
    @"."

let default_separator =
  let first_char str =
    if (UTF8.length str) > 0 then
      UTF8.get str 0
    else
      raise Not_found in
  (* first: try to use Barista-specific value *)
  try
    first_char (OS.get_env @"BARISTA_CLASSPATH_SEPARATOR")
  with _ ->
    (* second: try to use Java property *)
    try
      first_char (OS.get_property @"path.separator")
    with _ ->
      (* third: apply an heuristic *)
      if UTF8.contains @';' system_classpath then
        @';'
      else match Sys.os_type with
      | "Win32" -> @';'
      | "Unix" | "Cygwin" | _ -> @':'

let special_archive filename prefix =
  let len_prefix = UTF8.length prefix in
  let arch = ArchiveFile.make_of_path filename in
  let compute_prefixes () =
    let res =
      ArchiveFile.fold_entries
        (fun acc entry ->
          let entry_name = ArchiveEntry.get_filename entry in
          if UTF8.starts_with prefix entry_name then begin
            try
              let idx_slash = UTF8.index_from entry_name len_prefix @'/' in
              let jar_name = UTF8.substring entry_name len_prefix (pred idx_slash) in
              UTF8.Set.add jar_name acc
            with _ ->
              acc
          end else
            acc)
        UTF8.Set.empty
        arch in
    UTF8.Set.elements res in
  { big_archive_prefix = prefix;
    big_archive_file = arch;
    big_archive_prefix_list = lazy (compute_prefixes ()); }

let sym_archive_prefix = @"META-INF/sym/"

let web_archive_prefix = @"WEB-INF/classes"

let make_entry_of_path source path =
  let path = Path.absolute path in
  let fail_not_exists () =
    Error { entry_source = source;
            entry_path = path;
            entry_value = Does_not_exist path; } in
  let fail_archive () =
    Error { entry_source = source;
            entry_path = path;
            entry_value = Unable_to_open_archive path; } in
  try
    if Path.is_directory path then
      Directory { entry_source = source;
                  entry_path = path;
                  entry_value = (); }
    else if Path.exists path then begin
      if Path.check_suffix @".sym" path then
        try
          Big_archive { entry_source = source;
                        entry_path = path;
                        entry_value = special_archive path sym_archive_prefix; }
        with _ ->
          fail_archive ()
      else if Path.check_suffix @".war" path then
        try
          let big_archive = { big_archive_prefix = web_archive_prefix;
                              big_archive_file = ArchiveFile.make_of_path path;
                              big_archive_prefix_list = lazy [ @"" ]; } in
          Big_archive { entry_source = source;
                        entry_path = path;
                        entry_value = big_archive; }
        with _ ->
          fail_archive ()
      else
        try
          Archive { entry_source = source;
                    entry_path = path;
                    entry_value = ArchiveFile.make_of_path path; }
        with _ ->
          fail_archive ()
    end else
      fail_not_exists ()
  with Sys_error _ ->
    fail_not_exists ()

let make_entry_of_utf8 source str =
  str
  |> Path.make_of_utf8
  |> make_entry_of_path source

let map_path_elements source l =
  List.map (fun x -> make_entry_of_path source x) l

let map_utf8_elements source l =
  List.map (fun x -> make_entry_of_utf8 source x) l

let check_elements l =
  List.iter
    (function
      | Directory _ | Archive _ | Big_archive _ ->
          ()
      | Error { entry_source = Explicit; entry_value = err; _ } ->
          fail err
      | Error _ ->
          ())
    l;
  l

let make_of_utf8 ?(separator = default_separator) str =
  str
  |> UTF8.split_quotes separator
  |> map_utf8_elements Explicit
  |> check_elements
  
let make_of_path_list l =
  l
  |> map_path_elements Explicit
  |> check_elements

let make_of_utf8_list l =
  l
  |> map_utf8_elements Explicit
  |> check_elements

let java_home_utility = "/usr/libexec/java_home"

let get_core_libraries () =
  try
    let home =
      if Path.exists (Path.make_of_string java_home_utility) then begin
        try
          List.hd (OS.execute java_home_utility)
        with _ ->
          OS.get_env @"JAVA_HOME"
      end else
        OS.get_env @"JAVA_HOME" in
    let home = Path.make_of_utf8 home in
    let in_home str =
      str
      |> Path.make_of_utf8
      |> Path.concat home in
    let candidates =
      [ in_home @"/lib/ct.sym" ; (* JDK archive *)
        in_home @"/lib/rt.jar" ; (* JRE archive *)
        in_home @"/jre/lib/jfxrt.jar" ; (* JavaFX archive (inside JDK) *)
        in_home @"/lib/jfxrt.jar" ] (* JavaFX archive (inside JRE) *) in
    List.filter Path.exists candidates
  with _ ->
    []

let get_system_classpath_elements () =
  UTF8.split_quotes default_separator system_classpath

let get_default () =
  (get_core_libraries ())
  @ (List.map Path.make_of_utf8 @@ get_system_classpath_elements ())

let make () =
  (map_path_elements Core_library @@ get_core_libraries ())
  @ (map_utf8_elements System_classpath @@ get_system_classpath_elements ())

let make_empty () =
  []

let append ?(separator = default_separator) str cp =
  cp @ (make_of_utf8 ~separator str)

let append_path path cp =
  cp @ (check_elements @@ map_path_elements Explicit [path])

let prepend ?(separator = default_separator) str cp =
  (make_of_utf8 ~separator str) @ cp

let prepend_path path cp =
  (check_elements @@ map_path_elements Explicit [path]) @ cp


(* Functions *)

let open_entry arch str =
  ArchiveFile.find_entry arch str
  |> ArchiveFile.stream_of_entry arch

let rec lookup_with_prefix arch global_prefix str = function
  | prefix :: tl ->
      (try
        let full_name = global_prefix ++ prefix ++ @"/" ++ str in
        open_entry arch full_name
      with _ ->
        lookup_with_prefix arch global_prefix str tl)
  | [] ->
      fail (Class_not_found str)
  
let rec lookup cp str =
  match cp with
  | (Directory { entry_path = dir; _}) :: tl ->
      (try
        let path = Path.make_of_utf8 str in
        InputStream.make_of_path (Path.concat dir path)
      with _ ->
        lookup tl str)
  | (Archive { entry_value = arch; _ }) :: tl ->
      (try
        open_entry arch str
      with _ ->
        lookup tl str)
  | (Big_archive { entry_value = { big_archive_prefix;
                                   big_archive_file;
                                   big_archive_prefix_list; }; _ }) :: tl ->
      (try
        lookup_with_prefix
          big_archive_file
          big_archive_prefix
          str
          (Lazy.force big_archive_prefix_list)
      with _ ->
        lookup tl str)
  | (Error _) :: tl ->
      lookup tl str
  | [] ->
      fail (Class_not_found str)

let replace_last_slash_with_dollar str =
  try
    let len = UTF8.length str in
    let idx_slash = UTF8.rindex_from str (pred len) @'/' in
    let buff = UTF8Buffer.make_of_size len in
    UTF8Buffer.add_string buff (UTF8.substring str 0 (pred idx_slash));
    UTF8Buffer.add_char buff @'$';
    UTF8Buffer.add_string buff (UTF8.substring str (succ idx_slash) (pred len));
    UTF8Buffer.contents buff
  with Not_found ->
    str

let rec search_stream cp str =
  try
    lookup cp str
  with _ ->
    let str' = replace_last_slash_with_dollar str in
    if UTF8.equal str str' then
      fail (Class_not_found str)
    else
      search_stream cp str'

let open_stream cp str =
  let str = (UTF8.replace @'.' @'/' str) ++ @".class" in
  try
    search_stream cp str
  with _ ->
    fail (Class_not_found str)

let close cp =
  let close_entry = function
    | Directory _ | Error _ ->
        ()
    | Archive { entry_value; _ } ->
        ArchiveFile.close entry_value
    | Big_archive { entry_value = { big_archive_file; _ }; _ } ->
        ArchiveFile.close big_archive_file in
  List.iter close_entry cp

let close_noerr cp =
  try
    close cp
  with _ ->
    ()

let equal_entry e1 e2 =
  e1 = e2

let compare_entry e1 e2 =
  Pervasives.compare e1 e2

let hash_entry = function
  | Directory { entry_path; _ } -> Path.hash entry_path
  | Archive { entry_path; _ } -> 1 + (Path.hash entry_path)
  | Big_archive { entry_path; _ } -> 2 + (Path.hash entry_path)
  | Error { entry_path; _ } -> 3 + (Path.hash entry_path)

let equal cp1 cp2 =
  Utils.list_equal equal_entry cp1 cp2

let compare cp1 cp2 =
  Utils.list_compare compare_entry cp1 cp2

let hash cp =
  Utils.list_hash hash_entry cp

let to_utf8 cp =
  UTF8.concat_sep_map
    (UTF8.of_uchar default_separator)
    (function
      | Directory { entry_path; _ }
      | Archive { entry_path; _ }
      | Big_archive { entry_path; _ }
      | Error { entry_path; _ } ->
          Path.to_utf8 entry_path)
    cp

let to_internal_utf8 cp =
  UTF8.concat_sep_map
    @", "
    (function
      | Directory { entry_path; _ } ->
          @"Directory " ++ (Path.to_utf8 entry_path)
      | Archive { entry_path; _ } ->
          @"Archive " ++ (Path.to_utf8 entry_path)
      | Big_archive { entry_path; _ } ->
          @"Big_archive " ++ (Path.to_utf8 entry_path)
      | Error { entry_value = err; _ } ->
          @"Error " ++ (UTF8.of_string (string_of_error err)))
    cp
