(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

IFNDEF USE_JDK THEN

type t = Zip.entry

let get_filename entry =
  entry.Zip.filename
  |> UTF8.of_string

let is_directory entry =
  entry.Zip.is_directory

let get_extra entry =
  entry.Zip.extra
  |> Bytes.make_of_string

let get_comment entry =
  entry.Zip.comment
  |> UTF8.of_string

ELSE (* USE_JDK *)

type t = java'util'zip'ZipEntry java_instance

external utf8_of_java_string : java'lang'String java_instance -> UTF8.t =
  "%identity"

external bytes_of_byte_array : int JavaByteArray.t -> Bytes.t =
  "%identity"

let get_filename entry =
  entry
  |> Java.call "java.util.zip.ZipEntry.getName()"
  |> utf8_of_java_string

let is_directory entry =
  entry
  |> Java.call "java.util.zip.ZipEntry.isDirectory()"

let get_extra entry =
  let res =
    entry
    |> Java.call "java.util.zip.ZipEntry.getExtra()" in
  if JavaByteArray.is_null res then
    Bytes.empty
  else
    res
    |> bytes_of_byte_array

let get_comment entry =
  let res =
    entry
    |> Java.call "java.util.zip.ZipEntry.getComment()" in
  (if Java.is_null res then
    Java.make "String()" ()
  else
    res)
  |> utf8_of_java_string

END
