(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


IFNDEF USE_JDK THEN

(* Types *)

type t = {
    read_u1 : unit -> int;
    read_bytes : int -> Bytes.t;
    read_bytes_into : int -> Bytes.t -> int -> unit;
    read_available_bytes : int -> Bytes.t -> int -> int;
    close : unit -> unit;
  }


(* Exception *)

BARISTA_ERROR =
  | Unable_to_open_stream of (path : Path.t) ->
      Printf.sprintf "unable to open input stream (%S)"
        (Path.to_string path)
  | End_of_input_stream ->
      "end of input stream"
  | Unable_to_read_data ->
      "unable to read data"
  | Unable_to_close_stream ->
      "unable to close stream"
  | Data_is_too_large ->
      "data is too large"


(* Constructors *)

external string_of_bytes : Bytes.t -> string =
  "%identity"

external bytes_of_string : string -> Bytes.t =
  "%identity"

external string_of_path : Path.t -> string =
  "%identity"

let make_of_bytes str =
  let str = string_of_bytes str in
  let opened = ref true in
  let ptr = ref 0 in
  let read_u1 () =
    if !opened then
      let tmp = !ptr in
      if (tmp < String.length str) then
        let res = int_of_char str.[tmp] in
        ptr := tmp + 1;
        res
      else
        fail End_of_input_stream
    else
      fail Unable_to_read_data in
  let read_bytes nb =
    if !opened then
      try
        let tmp = !ptr in
        let res = String.sub str tmp nb in
        ptr := tmp + nb;
        bytes_of_string res
      with Invalid_argument _ ->
        fail End_of_input_stream
    else
      fail Unable_to_read_data in
  let read_bytes_into nb dst idx =
    if !opened then
      let dst = string_of_bytes dst in
      try
        let tmp = !ptr in
        String.blit str tmp dst idx nb;
        ptr := tmp + nb
      with Invalid_argument _ ->
        fail End_of_input_stream
    else
      fail Unable_to_read_data in
  let read_available_bytes nb dst idx =
    if !opened then
      let dst = string_of_bytes dst in
      try
        let tmp = !ptr in
        let len = String.length str in
        let nb = Utils.min_int nb (len - tmp) in
        String.blit str tmp dst idx nb;
        ptr := tmp + nb;
        nb
      with _ ->
        fail Unable_to_read_data
    else
      fail Unable_to_read_data in
  let close () =
    opened := false in
  { read_u1; read_bytes; read_bytes_into; read_available_bytes; close }

let make_of_channel chan =
  set_binary_mode_in chan true;
  let read_u1 () =
    try
      input_byte chan
    with
    | End_of_file -> fail End_of_input_stream
    | _ -> fail Unable_to_read_data in
  let read_bytes nb =
    try
      let res = String.create nb in
      really_input chan res 0 nb;
      bytes_of_string res
    with
    | Invalid_argument _ -> fail End_of_input_stream
    | End_of_file -> fail End_of_input_stream
    | _ -> fail Unable_to_read_data in
  let read_bytes_into nb dst idx =
    let dst = string_of_bytes dst in
    try
      really_input chan dst idx nb
    with
    | Invalid_argument _ -> fail End_of_input_stream
    | End_of_file -> fail End_of_input_stream
    | _ -> fail Unable_to_read_data in
  let read_available_bytes nb dst idx =
    let dst = string_of_bytes dst in
    try
      input chan dst idx nb
    with _ ->
      fail Unable_to_read_data in
  let close () =
    try
      close_in chan
    with _ ->
      fail Unable_to_close_stream in
  { read_u1; read_bytes; read_bytes_into; read_available_bytes; close }

let make_of_path path =
  try
    path
    |> string_of_path
    |> open_in
    |> make_of_channel
  with _ ->
    fail (Unable_to_open_stream path)


(* Casts, to avoid bound checking *)

external unsafe_u1 : int -> Utils.u1 =
  "%identity"

external unsafe_u2 : int -> Utils.u2 =
  "%identity"

external unsafe_u4 : int64 -> Utils.u4 =
  "%identity"

external unsafe_s1 : int -> Utils.s1 =
  "%identity"

external unsafe_s2 : int -> Utils.s2 =
  "%identity"

external unsafe_s4 : int32 -> Utils.s4 =
  "%identity"

external unsafe_s8 : int64 -> Utils.s8 =
  "%identity"


(* Functions *)

let read_u1 stream =
  unsafe_u1 (stream.read_u1 ())

let read_u2 stream =
  let a = stream.read_u1 () in
  let b = stream.read_u1 () in
  unsafe_u2 ((a lsl 8) lor b)

let read_u4 stream =
  let open Int64 in
  let a = of_int (stream.read_u1 ()) in
  let b = of_int (stream.read_u1 ()) in
  let c = of_int (stream.read_u1 ()) in
  let d = of_int (stream.read_u1 ()) in
  unsafe_u4 (logor
               (shift_left a 24)
               (logor
                  (shift_left b 16)
                  (logor (shift_left c 8) d)))

let read_s1 stream =
  let x = stream.read_u1 () in
  unsafe_s1 (if x < 128 then x else x - 256)

let read_s2 stream =
  let a = stream.read_u1 () in
  let b = stream.read_u1 () in
  let x = (a lsl 8) lor b in
  unsafe_s2 (if x < 32768 then x else x - 65536)

let read_s4 stream =
  let open Int32 in
  let a = of_int (stream.read_u1 ()) in
  let b = of_int (stream.read_u1 ()) in
  let c = of_int (stream.read_u1 ()) in
  let d = of_int (stream.read_u1 ()) in
  unsafe_s4 (logor
               (shift_left a 24)
               (logor
                  (shift_left b 16)
                  (logor (shift_left c 8) d)))

let read_s8 stream =
  let open Int64 in
  let a = of_int (stream.read_u1 ()) in
  let b = of_int (stream.read_u1 ()) in
  let c = of_int (stream.read_u1 ()) in
  let d = of_int (stream.read_u1 ()) in
  let e = of_int (stream.read_u1 ()) in
  let f = of_int (stream.read_u1 ()) in
  let g = of_int (stream.read_u1 ()) in
  let h = of_int (stream.read_u1 ()) in
  unsafe_s8 (logor
               (shift_left a 56)
               (logor
                  (shift_left b 48)
                  (logor
                     (shift_left c 40)
                     (logor
                        (shift_left d 32)
                        (logor
                           (shift_left e 24)
                           (logor
                              (shift_left f 16)
                              (logor
                                 (shift_left g 8) h)))))))

let read_bytes stream nb =
  stream.read_bytes nb

let read_bytes_into stream nb dst idx =
  stream.read_bytes_into nb dst idx

let read_available_bytes stream nb dst idx =
  stream.read_available_bytes nb dst idx

let read_utf8 stream =
  (read_u2 stream :> int)
  |> read_bytes stream
  |> UTF8.of_bytes

let read_elements stream f =
  let nb = read_u2 stream in
  let res = ref [] in
  for _i = 1 to (nb :> int) do
    let elem = f stream in
    res := elem :: !res
  done;
  List.rev !res

let close stream =
  stream.close ()

let close_noerr stream =
  try
    stream.close ()
  with _ ->
    ()

let try_with stream f =
  Utils.try_finally stream f close_noerr


(* Predefined stream *)

let stdin = make_of_channel stdin

ELSE (* USE_JDK *)

(* Types *)

type t = java'io'DataInputStream java_instance


(* Exception *)

BARISTA_ERROR =
  | Unable_to_open_stream of (path : Path.t) ->
      Printf.sprintf "unable to open input stream (%S)"
        (Path.to_string path)
  | End_of_input_stream ->
      "end of input stream"
  | Unable_to_read_data ->
      "unable to read data"
  | Unable_to_close_stream ->
      "unable to close stream"
  | Data_is_too_large ->
      "data is too large"


(* Constructors *)

external bytes_of_byte_array : int JavaByteArray.t -> Bytes.t =
  "%identity"

external byte_array_of_bytes : Bytes.t -> int JavaByteArray.t =
  "%identity"

external input_stream_of_in_channel : in_channel -> java'io'InputStream java_instance =
  "ocamljava_input_stream_of_in_channel"

external file_of_path : Path.t -> java'io'File java_instance =
  "%identity"

let make_of_bytes bytes =
  bytes
  |> byte_array_of_bytes
  |> Java.make "java.io.ByteArrayInputStream(_)"
  |> Java.make "java.io.DataInputStream(_)"

let make_of_channel chan =
  chan
  |> input_stream_of_in_channel
  |> Java.make "java.io.BufferedInputStream(_)"
  |> Java.make "java.io.DataInputStream(_)"

let make_of_path path =
  let file =
    path
    |> file_of_path in
  let files_state =
    Java.call "org.ocamljava.runtime.context.CurrentContext.getFilesState()" () in
  let hook =
    Java.call "org.ocamljava.runtime.context.FilesState.getFileHook()" files_state in
  let stream =
    if Java.is_not_null hook then
      Some (file
           |> Java.call "org.ocamljava.runtime.context.FilesState.resourceNameFromFile(_)" files_state
           |> Java.call "org.ocamljava.runtime.context.FileHook.getInputStream(_)" hook)
    else
      None in
  match stream with
  | Some st when Java.is_not_null st ->
      st
      |> Java.make "java.io.BufferedInputStream(_)"
      |> Java.make "java.io.DataInputStream(_)"
  | Some _ | None ->
      try
        file
        |> Java.make "java.io.FileInputStream(java.io.File)"
        |> Java.make "java.io.BufferedInputStream(_)"
        |> Java.make "java.io.DataInputStream(_)"
      with Java_exception _ ->
        fail (Unable_to_open_stream path)


(* Casts, to avoid bound checking *)

external unsafe_u1 : int32 -> Utils.u1 =
  "%int32_to_int"

external unsafe_u2 : int32 -> Utils.u2 =
  "%int32_to_int"

external unsafe_u4 : int64 -> Utils.u4 =
  "%identity"

external unsafe_s1 : int -> Utils.s1 =
  "%identity"

external unsafe_s2 : int -> Utils.s2 =
  "%identity"

external unsafe_s4 : int32 -> Utils.s4 =
  "%identity"

external unsafe_s8 : int64 -> Utils.s8 =
  "%identity"


(* Functions *)

let read_u1 stream =
  try
    Java.call "java.io.DataInputStream.readUnsignedByte()" stream
    |> unsafe_u1 
  with
  | Java_exception e when Java.instanceof "java.io.EOFException" e ->
      fail End_of_input_stream
  | Java_exception _ ->
      fail Unable_to_read_data

let read_u2 stream =
  try
    Java.call "java.io.DataInputStream.readUnsignedShort()" stream
    |> unsafe_u2
  with
  | Java_exception e when Java.instanceof "java.io.EOFException" e ->
      fail End_of_input_stream
  | Java_exception _ ->
      fail Unable_to_read_data

let read_u4 stream =
  try
    Java.call "java.io.DataInputStream.readInt()" stream
    |> Int64.of_int32
    |> Int64.logand 0x00000000FFFFFFFFL
    |> unsafe_u4
  with
  | Java_exception e when Java.instanceof "java.io.EOFException" e ->
      fail End_of_input_stream
  | Java_exception _ ->
      fail Unable_to_read_data

let read_s1 stream =
  try
    Java.call "java.io.DataInputStream.readByte()" stream
    |> unsafe_s1
  with
  | Java_exception e when Java.instanceof "java.io.EOFException" e ->
      fail End_of_input_stream
  | Java_exception _ ->
      fail Unable_to_read_data

let read_s2 stream =
  try
    Java.call "java.io.DataInputStream.readShort()" stream
    |> unsafe_s2
  with
  | Java_exception e when Java.instanceof "java.io.EOFException" e ->
      fail End_of_input_stream
  | Java_exception _ ->
      fail Unable_to_read_data

let read_s4 stream =
  try
    Java.call "java.io.DataInputStream.readInt()" stream
    |> unsafe_s4
  with
  | Java_exception e when Java.instanceof "java.io.EOFException" e ->
      fail End_of_input_stream
  | Java_exception _ ->
      fail Unable_to_read_data

let read_s8 stream =
  try
    Java.call "java.io.DataInputStream.readLong()" stream
    |> unsafe_s8
  with
  | Java_exception e when Java.instanceof "java.io.EOFException" e ->
      fail End_of_input_stream
  | Java_exception _ ->
      fail Unable_to_read_data

let read_bytes stream nb =
  try
    let res = Java.make_array "byte[]" (Int32.of_int nb) in
    Java.call "java.io.DataInputStream.readFully(_)" stream res;
    bytes_of_byte_array res
  with
  | Java_exception e when Java.instanceof "java.io.EOFException" e ->
      fail End_of_input_stream
  | Java_exception _ ->
      fail Unable_to_read_data

let read_bytes_into stream nb dst idx =
  let nb = Int32.of_int nb in
  let res =
    try
      Java.call "java.io.DataInputStream.read(_,_,_)"
        stream
        (byte_array_of_bytes dst)
        (Int32.of_int idx)
        nb
    with Java_exception _ ->
      fail Unable_to_read_data in
  if res < nb then fail End_of_input_stream

let read_available_bytes stream nb dst idx =
  let dst = byte_array_of_bytes dst in
  try
    Java.call "java.io.DataInputStream.read(_,_,_)"
      stream
      dst
      (Int32.of_int idx)
      (Int32.of_int nb)
    |> Int32.to_int
  with Java_exception _ ->
    fail Unable_to_read_data

external utf8_of_java_string : java'lang'String java_instance -> UTF8.t =
  "%identity"

let read_utf8 stream =
  Java.make "java.io.DataInputStream(_)" stream
  |> Java.call "java.io.DataInputStream.readUTF()"
  |> utf8_of_java_string

let read_elements stream f =
  let nb = read_u2 stream in
  let res = ref [] in
  for _i = 1 to (nb :> int) do
    let elem = f stream in
    res := elem :: !res
  done;
  List.rev !res

let close stream =
  try
    Java.call "java.io.DataInputStream.close()" stream
  with Java_exception _ ->
    fail Unable_to_close_stream

let close_noerr stream =
  try
    Java.call "java.io.DataInputStream.close()" stream
  with Java_exception _ ->
    ()

let try_with stream f =
  Utils.try_finally stream f close_noerr


(* Predefined stream *)

let stdin = make_of_channel stdin

END
