(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


module type T = sig

  val names : UTF8.t list

  val description : UTF8.t

  val latex_description : UTF8.t list

  type parameters

  val make_parameters : unit -> parameters

  val switches_of_parameters : parameters -> UArg.switch list

  val others_of_parameters : parameters -> UArg.others option

  val run : parameters -> unit

end

class no_parameter = object

  method switches : UArg.switch list =
    []

  method others : UArg.others option =
    None

end

class base_parameters elements_desc = object (self)

  val mutable class_path = []

  val mutable class_path_reset = false

  val mutable elements = []

  method add_class_path x =
    class_path <- x :: class_path

  method set_class_path x =
    class_path_reset <- true;
    class_path <- [ x ]

  method build_class_path =
    let init =
      if class_path_reset then
        ClassPath.make_empty ()
      else
        ClassPath.make () in
    List.fold_right
      (fun elem acc ->
        ClassPath.append elem acc)
      class_path
      init

  method add_element x =
    elements <- x :: elements

  method get_elements =
    List.rev elements

  method switches =
    let open UArg in
    [ { identifier = @"-cp";
        kind = String { switch_desc = @"<path>";
                        handle_string = self#add_class_path };
        documentation = @"Add to classpath"; } ;
      { identifier = @"-classpath";
        kind = String { switch_desc = @"<path>";
                        handle_string = self#set_class_path };
        documentation = @"Set classpath"; } ]

  method others =
    let open UArg in
    Some ({ argument_desc = elements_desc;
            handle_argument = self#add_element })
end

let graph_formats =
  List.map
    (fun x -> (Graph.identifier_of_format x), x)
    Graph.all_formats

class graph_format = object (self)

  val mutable graph_format = Graph.Dot

  method set_graph_format gf =
    graph_format <- gf

  method get_graph_format =
    graph_format

  method switches =
    let open UArg in
    [ { identifier = @"-format";
        kind = Choice { switch_options = List.map fst graph_formats;
                        handle_choice = (fun str ->
                          self#set_graph_format (Utils.list_assoc UTF8.equal str graph_formats)); };
        documentation = @"Graph format"; } ]

  method others : UArg.others option =
    None

end

let parameters_functions constr =
  constr, (fun inst -> inst#switches), (fun inst -> inst#others)
