(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

let (++) = UTF8.(++)

type version =
  | Version_1_0

let utf8_of_version = function
  | Version_1_0 -> @"Manifest-Version: 1.0"

type custom_creator = {
    creator_name : UTF8.t;
    creator_version : UTF8.t;
  }

type creator =
  | Barista_creator
  | Custom_creator of custom_creator
  | No_creator

let lines_of_creator = function
  | Barista_creator ->
      let version = UTF8.of_string CurrentVersion.value in
      [ SPRINTF ("Created-By: %s (Barista)" version) ]
  | Custom_creator { creator_name; creator_version } ->
      [ SPRINTF ("Created-By: %s (%s)" creator_version creator_name) ]
  | No_creator ->
      []

type required_extension = {
    rext_short_name : UTF8.t;
    rext_full_name : UTF8.t;
    rext_spec_vendor : UTF8.t;
    rext_spec_version : UTF8.t;
    rext_impl_vendor_id : UTF8.t;
    rext_impl_vendor : UTF8.t;
    rext_impl_version : UTF8.t;
    rext_impl_url : UTF8.t;
  }

let lines_of_required_extension ext =
  let line key value =
    SPRINTF ("%s-%s: %s" ext.rext_short_name key value) in
  [ line @"Extension-Name"           ext.rext_full_name ;
    line @"Specification-Vendor"     ext.rext_spec_vendor ;
    line @"Specification-Version"    ext.rext_spec_version ;
    line @"Implementation-Vendor-Id" ext.rext_impl_vendor_id ;
    line @"Implementation-Vendor"    ext.rext_impl_vendor ;
    line @"Implementation-Version"   ext.rext_impl_version ;
    line @"Implementation-URL"       ext.rext_impl_url ]

type provided_extension = {
    pext_name : UTF8.t;
    pext_impl_title : UTF8.t;
    pext_impl_version : UTF8.t;
    pext_impl_vendor : UTF8.t;
    pext_impl_vendor_id : UTF8.t;
    pext_spec_title : UTF8.t;
    pext_spec_version : UTF8.t;
    pext_spec_vendor : UTF8.t;
  }

let lines_of_provided_extension ext =
  let line key value =
    SPRINTF ("%s: %s" key value) in
  match ext with
  | Some ext ->
      [ line @"Extension-Name"           ext.pext_name ;
        line @"Implementation-Title"     ext.pext_impl_title ;
        line @"Implementation-Version"   ext.pext_impl_version ;
        line @"Implementation-Vendor"    ext.pext_impl_vendor ;
        line @"Implementation-Vendor-Id" ext.pext_impl_vendor_id ;
        line @"Specification-Title"      ext.pext_spec_title ;
        line @"Specification-Version"    ext.pext_spec_version ;
        line @"Specification-Vendor"     ext.pext_spec_vendor ]
  | None ->
      []

type entry = {
    entry_path : UTF8.t;
    entry_type : UTF8.t option;
    entry_bean : bool option;
    entry_sealed : bool;
  }

let lines_of_type = function
  | Some typ -> [ @"Content-Type: " ++ typ ]
  | None -> []

let lines_of_bean = function
  | Some false -> [ @"Java-Bean: false" ]
  | Some true -> [ @"Java-Bean: true" ]
  | None -> []

let lines_of_sealed = function
  | true -> [ @"Sealed: true" ]
  | false -> []

let lines_of_entry ent =
  [  @"Name: " ++ ent.entry_path ]
  @ (lines_of_type ent.entry_type)
  @ (lines_of_bean ent.entry_bean)
  @ (lines_of_sealed ent.entry_sealed)

let lines_of_entry_list l =
  List.map lines_of_entry l
  |> List.concat

type t = {
    version : version;
    creator : creator;
    sealed : bool;
    main_class : Name.for_class option;
    class_path : UTF8.t list;
    extensions : required_extension list;
    extension : provided_extension option;
    entries : entry list;
  }

let path_in_archive = @"META-INF/MANIFEST.MF"

let default =
  { version = Version_1_0 ;
    creator = Barista_creator ;
    sealed = false ;
    main_class = None ;
    class_path = [] ;
    extensions = [] ;
    extension = None ;
    entries = [] ; }

let lines_of_main_class = function
  | Some cn ->
      [ @"Main-Class: " ++ (Name.external_utf8_for_class cn) ]
  | None ->
      []

let lines_of_class_path = function
  | (_ :: _) as l ->
      [ @"Class-Path: " ++  (UTF8.concat_sep @" " l) ]
  | [] ->
      []

let lines_of_extensions = function
  | (_ :: _) as l ->
      let ext_names =
        UTF8.concat_sep_map @" " (fun re -> re.rext_short_name) l in
      ((UTF8.of_string "Extension-List: ") ++ ext_names)
      :: (List.concat (List.map lines_of_required_extension l))
  | [] ->
      []

let to_lines manifest =
  [ utf8_of_version manifest.version ]
  @ (lines_of_creator manifest.creator)
  @ (lines_of_sealed manifest.sealed)
  @ (lines_of_main_class manifest.main_class)
  @ (lines_of_class_path manifest.class_path)
  @ (lines_of_extensions manifest.extensions)
  @ (lines_of_provided_extension manifest.extension)
  @ (lines_of_entry_list manifest.entries)

let to_utf8 manifest =
  (manifest
  |> to_lines
  |> UTF8.concat_sep @"\n") ++ @"\n"
