(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


(* Types *)

type value =
  | Unknown_value
  | Unknown_integer_value
  | Known_integer_value of int32
  | Unknown_float_value
  | Known_float_value of float
  | Unknown_long_value
  | Known_long_value of int64
  | Unknown_double_value
  | Known_double_value of float
  | Null_value
  | Object_value

(* Float values are compared with '=', only possibly resulting in missed
   opportunities for optimizations. *)
let equal_value v1 v2 =
  match v1, v2 with
  | Unknown_value, Unknown_value -> true
  | Unknown_integer_value, Unknown_integer_value -> true
  | Known_integer_value x, Known_integer_value y -> x = y
  | Unknown_float_value, Unknown_float_value -> true
  | Known_float_value x, Known_float_value y -> x = y
  | Unknown_long_value, Unknown_long_value -> true
  | Known_long_value x, Known_long_value y -> x = y
  | Unknown_double_value, Unknown_double_value -> true
  | Known_double_value x, Known_double_value y -> x = y
  | Null_value, Null_value -> true
  | Object_value, Object_value -> true
  | _ -> false

let java_lang_Object = Name.make_for_class_from_external @"java.lang.Object"

let verification_type_info_of_value = function
  | Unknown_value -> Attribute.Top_variable_info
  | Unknown_integer_value
  | Known_integer_value _ -> Attribute.Integer_variable_info
  | Unknown_float_value
  | Known_float_value _ -> Attribute.Float_variable_info
  | Unknown_long_value
  | Known_long_value _ -> Attribute.Long_variable_info
  | Unknown_double_value
  | Known_double_value _ -> Attribute.Double_variable_info
  | Null_value -> Attribute.Null_variable_info
  | Object_value -> Attribute.Object_variable_info (`Class_or_interface java_lang_Object)

type local =
  | Local of value
  | Place_holder

type locals = local array

type stack = value list (* stack top is list head *)

type t = {
    locals : locals;
    stack : stack;
  }

type operation =
  | Pop
  | Pop2
  | Push_integer of int32 * (Utils.u2 option)
  | Push_float of float * (Utils.u2 option)
  | Push_long of int64 * (Utils.u2 option)
  | Push_double of float * (Utils.u2 option)
  | Push_null of Utils.u2 option
  | Execute of Instruction.t
  | Sequence of operation list

let rec instruction_list_of_operation = function
  | Pop -> [ Instruction.POP ]
  | Pop2 -> [ Instruction.POP2 ]
  | Push_integer (_, Some idx) ->
      (match (idx :> int) with
      | 0 -> [ Instruction.ILOAD_0 ]
      | 1 -> [ Instruction.ILOAD_1 ]
      | 2 -> [ Instruction.ILOAD_2 ]
      | 3 -> [ Instruction.ILOAD_3 ]
      | i when i < 256 -> [ Instruction.ILOAD (Utils.u1 i) ]
      | _ -> [ Instruction.WIDE_ILOAD idx ])
  | Push_integer (-1l, _) -> [ Instruction.ICONST_M1 ]
  | Push_integer (0l, _) -> [ Instruction.ICONST_0 ]
  | Push_integer (1l, _) -> [ Instruction.ICONST_1 ]
  | Push_integer (2l, _) -> [ Instruction.ICONST_2 ]
  | Push_integer (3l, _) -> [ Instruction.ICONST_3 ]
  | Push_integer (4l, _) -> [ Instruction.ICONST_4 ]
  | Push_integer (5l, _) -> [ Instruction.ICONST_5 ]
  | Push_integer (i, _) ->
      if (i >= -128l) && (i <= 127l) then
        [ Instruction.BIPUSH (Utils.s1 (Int32.to_int i)) ]
      else if (i >= -32768l) && (i <= 32767l) then
        [ Instruction.SIPUSH (Utils.s2 (Int32.to_int i)) ]
      else
        [ Instruction.LDC_W (`Int i) ]
  | Push_float (_, Some idx) ->
      (match (idx :> int) with
      | 0 -> [ Instruction.FLOAD_0 ]
      | 1 -> [ Instruction.FLOAD_1 ]
      | 2 -> [ Instruction.FLOAD_2 ]
      | 3 -> [ Instruction.FLOAD_3 ]
      | i when i < 256 -> [ Instruction.FLOAD (Utils.u1 i) ]
      | _ -> [ Instruction.WIDE_FLOAD idx ])
  | Push_float (0.0, _) -> [ Instruction.FCONST_0 ]
  | Push_float (1.0, _) -> [ Instruction.FCONST_1 ]
  | Push_float (2.0, _) -> [ Instruction.FCONST_2 ]
  | Push_float (f, _) -> [ Instruction.LDC_W (`Float f) ]
  | Push_long (_, Some idx) ->
      (match  (idx :> int) with
      | 0 -> [ Instruction.LLOAD_0 ]
      | 1 -> [ Instruction.LLOAD_1 ]
      | 2 -> [ Instruction.LLOAD_2 ]
      | 3 -> [ Instruction.LLOAD_3 ]
      | i when i < 256 -> [ Instruction.LLOAD (Utils.u1 i) ]
      | _ -> [ Instruction.WIDE_LLOAD idx ])
  | Push_long (-1L, _) -> [ Instruction.ICONST_M1; Instruction.I2L ]
  | Push_long (0L, _) -> [ Instruction.LCONST_0 ]
  | Push_long (1L, _) -> [ Instruction.LCONST_1 ]
  | Push_long (2L, _) -> [ Instruction.ICONST_2; Instruction.I2L ]
  | Push_long (3L, _) -> [ Instruction.ICONST_3; Instruction.I2L ]
  | Push_long (4L, _) -> [ Instruction.ICONST_4; Instruction.I2L ]
  | Push_long (5L, _) -> [ Instruction.ICONST_5; Instruction.I2L ]
  | Push_long (l, _) ->
      if (l >= -128L) && (l <= 127L) then
        [ Instruction.BIPUSH (Utils.s1 (Int64.to_int l)); Instruction.I2L ]
      else if (l >= -32768L) && (l <= 32767L) then
        [ Instruction.SIPUSH (Utils.s2 (Int64.to_int l)); Instruction.I2L ]
      else
        [ Instruction.LDC2_W (`Long l) ]
  | Push_double (_, Some idx) ->
      (match (idx :> int) with
      | 0 -> [ Instruction.DLOAD_0 ]
      | 1 -> [ Instruction.DLOAD_1 ]
      | 2 -> [ Instruction.DLOAD_2 ]
      | 3 -> [ Instruction.DLOAD_3 ]
      | i when i < 256 -> [ Instruction.DLOAD (Utils.u1 i) ]
      | _ ->  [ Instruction.WIDE_DLOAD idx ])
  | Push_double (0.0, _) -> [ Instruction.DCONST_0 ]
  | Push_double (1.0, _) -> [ Instruction.DCONST_1 ]
  | Push_double (d, _) -> [ Instruction.LDC2_W (`Double d) ]
  | Push_null (Some idx) ->
      (match (idx :> int) with
      | 0 -> [ Instruction.ALOAD_0 ]
      | 1 -> [ Instruction.ALOAD_1 ]
      | 2 -> [ Instruction.ALOAD_2 ]
      | 3 -> [ Instruction.ALOAD_3 ]
      | i when i < 256 -> [ Instruction.ALOAD (Utils.u1 i) ]
      | _ -> [ Instruction.WIDE_ALOAD idx ])
  | Push_null None -> [ Instruction.ACONST_NULL ]
  | Execute i -> [ i ]
  | Sequence l ->
      let l = List.map instruction_list_of_operation l in
      List.flatten l

let cat1_operation = function
  | Push_integer _
  | Push_float _
  | Push_null _ -> true
  | _ -> false

let cat2_operation = function
  | Push_long _
  | Push_double _ -> true
  | _ -> false

let rec operation_eq x y =
  match x, y with
  | Pop, Pop -> true
  | Pop2, Pop2 -> true
  | Push_integer (i1, li1), Push_integer (i2, li2) -> i1 = i2 && li1 = li2
  | Push_float (f1, li1), Push_float (f2, li2) -> f1 = f2 && li1 = li2
  | Push_long (l1, li1), Push_long (l2, li2) -> l1 = l2 && li1 = li2
  | Push_double (d1, li1), Push_double (d2, li2) -> d1 = d2 && li1 = li2
  | Push_null li1, Push_null li2 -> li1 = li2
  | Execute i1, Execute i2 -> Instruction.equal i1 i2
  | Sequence ol1, Sequence ol2 -> operation_list_eq ol1 ol2
  | _ -> false
and operation_list_eq l1 l2 =
  (l1 == l2)
  || (try
      List.for_all2 operation_eq l1 l2
  with Invalid_argument _ -> false)
and operation_list_eq' l1 l2 =
  (l1 == l2)
  || (try
      List.for_all2 (fun (_, x) (_, y) -> operation_eq x y) l1 l2
  with Invalid_argument _ -> false)

let rec remove_pops = function
  | (_, Pop)  :: (_, op)  :: tl when cat1_operation op -> remove_pops tl
  | (_, Pop2) :: (_, op)  :: tl when cat2_operation op -> remove_pops tl
  | (_, Pop2) :: (_, op1) :: (_, op2) :: tl when (cat1_operation op1) && (cat1_operation op2) -> remove_pops tl
  | (_, Pop2) :: (_, Execute Instruction.SWAP) :: (_, op1) :: (_, op2) :: tl when (cat1_operation op1) && (cat1_operation op2) -> remove_pops tl
  | (_, Pop) :: (_, Pop) :: (_, Execute Instruction.SWAP) :: (_, op1) :: (_, op2) :: tl when (cat1_operation op1) && (cat1_operation op2) -> remove_pops tl
  | hd :: tl -> hd :: (remove_pops tl)
  | [] -> []

let rec expand_sequence = function
  | Sequence l -> List.flatten (List.map expand_sequence l)
  | x -> [ x ]

let expand_sequences (l : (Utils.u2 * operation) list) =
  let l =
    List.map
      (fun (line, operation) ->
        let operations = expand_sequence operation in
        let lines = List.map (fun _ -> line) operations in
        List.combine lines operations)
      l in
  List.flatten l

let map_operations (l : (Utils.u2 * operation) list) =
  let l = expand_sequences l in
  let l = List.rev l in
  let l = Utils.fix_point operation_list_eq' remove_pops l in
  let l = List.rev l in
  let l =
    List.map
      (fun (line, operation) ->
        let instructions = instruction_list_of_operation operation in
        let lines = List.map (fun _ -> line) instructions in
        List.combine lines instructions)
      l in
  List.flatten l


(* Exception *)

BARISTA_ERROR =
  | Unsupported_instruction of (x : string) ->
      Printf.sprintf "unsupported instruction: %S" x
  | Empty_stack -> "empty stack"
  | Invalid_local_index of (i : Utils.u2) * (l : int) ->
      Printf.sprintf "invalid local index (%d, length %d)" (i :> int) l
  | Invalid_stack_top of (w : Attribute.verification_type_info) * (f : Attribute.verification_type_info) ->
      Printf.sprintf "invalid stack top: %S waited but %S found"
        (Attribute.string_of_verification_type_info w)
        (Attribute.string_of_verification_type_info f)
  | Invalid_local_contents of (i : Utils.u2) * (w : Attribute.verification_type_info) * (f : Attribute.verification_type_info) ->
      Printf.sprintf "invalid local contents at index %d: %S waited but %S found" 
        (i :> int)
        (Attribute.string_of_verification_type_info w)
        (Attribute.string_of_verification_type_info f)
  | Invalid_local_contents_placeholder of (i : Utils.u2) * (w : Attribute.verification_type_info) ->
      Printf.sprintf "invalid local contents at index %d: %S waited but placeholder found" 
        (i :> int)
        (Attribute.string_of_verification_type_info w)
  | Reference_waited of (f : Attribute.verification_type_info) ->
      Printf.sprintf "reference waited but %S found"
        (Attribute.string_of_verification_type_info f)
  | Reference_waited_placeholder_found -> "reference waited but placeholder found"
  | Array_waited -> "array waited"
  | Category1_waited -> "category1 waited"
  | Category2_waited -> "category2 waited"
  | Different_stack_sizes of (sz1 : int) * (sz2 : int) ->
      Printf.sprintf "different stack sizes (%d and %d)" sz1 sz2
  | Invalid_primitive_array_type -> "invalid primitive array type"


(* Construction *)

let make_empty () =
  { locals = [||]; stack = []; }

let value_of_constant_descriptor = function
  | `Int x -> Known_integer_value x
  | `Float x -> Known_float_value x
  | `String _ -> Object_value
  | `Class_or_interface _ -> Object_value
  | `Array_type _ -> Object_value
  | `Long x -> Known_long_value x
  | `Double x -> Known_double_value x
  | `Interface_method _ -> Object_value
  | `Method_type _ -> Object_value
  | `Method_handle _ -> Object_value

let of_list l =
  let l =
    List.map
      (fun elem ->
        match elem with
        | Unknown_long_value
        | Known_long_value _
        | Unknown_double_value 
        | Known_double_value _ ->
            [ Local elem; Place_holder ]
        | x -> [ Local x ])
      l in
  let l = List.concat l in
  Array.of_list l

let value_of_parameter_descriptor = function
  | `Boolean -> Unknown_integer_value
  | `Byte -> Unknown_integer_value
  | `Char -> Unknown_integer_value
  | `Double -> Unknown_double_value
  | `Float -> Unknown_float_value
  | `Int -> Unknown_integer_value
  | `Long -> Unknown_long_value
  | `Short -> Unknown_integer_value
  | `Class _ -> Object_value
  | `Array _ -> Object_value

let make_of_parameters inst l =
  let l = List.map value_of_parameter_descriptor l in
  let l = if inst then Object_value :: l else l in
  { locals = of_list l; stack = []; }

let make_of_method = function
  | Method.Regular { Method.flags; descriptor; _ } ->
      let l = fst descriptor in
      let l = List.map value_of_parameter_descriptor l in
      let l =
        if AccessFlag.mem_method `Static flags then
          l
        else
          Object_value :: l in
      { locals = of_list l; stack = [] }
  | Method.Constructor { Method.cstr_descriptor = l ; _ } ->
      let l = List.map value_of_parameter_descriptor l in
      let l = Object_value :: l in
      { locals = of_list l; stack = [] }
  | Method.Initializer _ ->
      make_empty ()


(* Access and modification *)

let locals_size st =
  Array.length st.locals

let stack_size st =
  List.fold_left
    (fun acc x ->
      acc +
        (match x with
        | Unknown_long_value
        | Known_long_value _
        | Unknown_double_value
        | Known_double_value _ -> 2
        | _ -> 1))
    0
    st.stack

let array_for_all2 ?(n = Utils.max_int_value) p a1 a2 =
  let i = ref (pred (Array.length a1)) in
  while (!i >= 0) && (!i < n) && (p a1.(!i) a2.(!i)) do
    decr i
  done;
  !i < 0

let same_local loc1 loc2 =
  match loc1, loc2 with
  | Local l1, Local l2 -> equal_value l1 l2
  | Place_holder, Place_holder ->
      true
  | _ -> false

let same_locals st1 st2 =
  ((locals_size st1) = (locals_size st2))
    && (array_for_all2 same_local st1.locals st2.locals)

let same_stack st1 st2 =
  ((stack_size st1) = (stack_size st2))
    && (List.for_all2 equal_value st1.stack st2.stack)

let equal st1 st2 =
  (st1 == st2)
|| ((same_locals st1 st2) && (same_stack st1 st2))

let push v s =
  v :: s

let push_local v s =
  match v with
  | Local v -> push v s
  | Place_holder -> assert false

let push_return_value x s =
  match x with
  | `Void -> s
  | #Descriptor.for_parameter as y ->
      push (value_of_parameter_descriptor y) s

let top = function
  | hd :: _ -> hd
  | [] -> fail Empty_stack

let pop = function
  | _ :: tl -> tl
  | [] -> fail Empty_stack

let pop_if v s =
  let v' = top s in
  let v_vti = verification_type_info_of_value v in
  let v'_vti = verification_type_info_of_value v' in
  let popable = match v_vti with
  | Attribute.Object_variable_info _ -> true
  | _ -> Attribute.equal_verification_type_info v_vti v'_vti in
  if popable then
    pop s
  else
    fail (Invalid_stack_top (v_vti, v'_vti))

let is_category1 = function
  | Unknown_value
  | Unknown_integer_value
  | Known_integer_value _
  | Unknown_float_value
  | Known_float_value _
  | Null_value
  | Object_value -> true
  | Unknown_long_value
  | Known_long_value _
  | Unknown_double_value
  | Known_double_value _ -> false

let pop_if_category1 = function
  | hd :: tl -> if is_category1 hd then hd, tl else fail Category1_waited
  | [] -> fail Empty_stack

let pop_if_cat2 = function
  | hd :: tl -> if not (is_category1 hd) then hd, tl else fail Category2_waited
  | [] -> fail Empty_stack

let empty () =
  []

let only_exception () =
  [Object_value]

let load i l =
  let j = (i : Utils.u2 :> int) in
  let len = Array.length l in
  if j >= 0 && j < len then
    l.(j)
  else
    fail (Invalid_local_index (i, len))

let check_load i l v =
  let v' = load i l in
  let v_vti = verification_type_info_of_value v in
  match v' with
  | Local v' ->
      let v'_vti = verification_type_info_of_value v' in
      if not (Attribute.equal_verification_type_info v_vti v'_vti) then
        fail (Invalid_local_contents (i, v_vti, v'_vti));
      v'
  | _ -> 
      fail (Invalid_local_contents_placeholder (i, v_vti))

let store i v l =
  let long_or_double = function
    | Unknown_long_value
    | Known_long_value _
    | Unknown_double_value
    | Known_double_value _ -> true
    | _ -> false in
  let long_or_double_local = function
    | Local l -> long_or_double l
    | _ -> false in
  let i = (i : Utils.u2 :> int) in
  let len = Array.length l in
  let is_long_or_double = long_or_double v in
  let sz = (succ i) + (if is_long_or_double then 1 else 0) in
  let l' =
    Array.init
      (Utils.max_int sz len)
      (fun i -> if i < len then l.(i) else Local Unknown_value) in
  l'.(i) <- Local v;
  if (not is_long_or_double) && (i < len) && (long_or_double_local l.(i)) then begin
    l'.(i + 1) <- Local Unknown_value
  end;
  if (i < len) && (i > 0) && (l.(i) = Place_holder) then begin
    l'.(pred i) <- Local Unknown_value
  end;
  if is_long_or_double then begin
    l'.(succ i) <- Place_holder
  end;
  l'


(* Operations *)

let check_reference x =
  match x with
  | Unknown_value
  | Unknown_integer_value
  | Known_integer_value _
  | Unknown_float_value
  | Known_float_value _
  | Unknown_long_value
  | Known_long_value _
  | Unknown_double_value
  | Known_double_value _ ->
      fail (Reference_waited (verification_type_info_of_value x))
  | Null_value
  | Object_value -> ()

let check_reference_local x =
  match x with
  | Local x -> check_reference x
  | Place_holder -> fail Reference_waited_placeholder_found

let value_of_array_element = function
  | `Array_type _ -> Object_value
  | `Class_or_interface _ -> Object_value

let value_of_array_primitive = function
  | `Boolean -> Object_value
  | `Char -> Object_value
  | `Float -> Object_value
  | `Double -> Object_value
  | `Byte -> Object_value
  | `Short -> Object_value
  | `Int -> Object_value
  | `Long -> Object_value
  | _ -> fail Invalid_primitive_array_type

let update i ev =
  let locals = Array.copy ev.locals in
  let stack = ev.stack in
  let operation_of_reference idx = function
    | Local Null_value -> Push_null (Some idx)
    | Local Object_value -> Execute i
    | _ -> assert false in
  let operation_of_double idx = function
    | Unknown_double_value -> Execute i
    | Known_double_value x -> Push_double (x, Some idx)
    | _ -> assert false in
  let operation_of_float idx = function
    | Unknown_float_value -> Execute i
    | Known_float_value x -> Push_float (x, Some idx)
    | _ -> assert false in
  let operation_of_integer idx = function
    | Unknown_integer_value -> Execute i
    | Known_integer_value x -> Push_integer (x, Some idx)
    | _ -> assert false in
  let operation_of_long idx = function
    | Unknown_long_value -> Execute i
    | Known_long_value x -> Push_long (x, Some idx)
    | _ -> assert false in
  let operation_of_constant_descriptor = function
  | `Int x -> Push_integer (x, None)
  | `Float x -> Push_float (x, None)
  | `Long x -> Push_long (x, None)
  | `Double x -> Push_double (x, None)
  | `String _
  | `Class_or_interface _
  | `Array_type _
  | `Interface_method _
  | `Method_type _
  | `Method_handle _ -> Execute i in
  let operation_of_value = function
  | Known_integer_value x -> Push_integer (x, None)
  | Known_float_value x -> Push_float (x, None)
  | Known_long_value x -> Push_long (x, None)
  | Known_double_value x -> Push_double (x, None)
  | Unknown_value
  | Unknown_integer_value
  | Unknown_float_value
  | Unknown_long_value
  | Unknown_double_value
  | Null_value
  | Object_value -> Execute i in
  match i with
  | Instruction.AALOAD ->
      let stack = pop_if Unknown_integer_value stack in
      let topv = top stack in
      check_reference topv;
      let stack = pop stack in
      let stack =
        (match topv with
        | Null_value -> push Object_value stack
        | Object_value -> push Object_value stack
        | _ -> fail Array_waited) in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.AASTORE ->
      let topv = top stack in
      check_reference topv;
      let stack = pop stack in
      let stack = pop_if Unknown_integer_value stack in
      let topv = top stack in
      check_reference topv;
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ACONST_NULL ->
      let stack = push Null_value stack in
      { locals = locals; stack = stack; }, Push_null None
  | Instruction.ALOAD parameter ->
      let loc = load (Utils.u2_of_u1 parameter) locals in
      check_reference_local loc;
      let stack = push_local loc stack in
      { locals = locals; stack = stack; }, (operation_of_reference (Utils.u2_of_u1 parameter) loc)
  | Instruction.ALOAD_0 ->
      let loc = load (Utils.u2 0) locals in
      check_reference_local loc;
      let stack = push_local loc stack in
      { locals = locals; stack = stack; }, (operation_of_reference (Utils.u2 0) loc)
  | Instruction.ALOAD_1 ->
      let loc = load (Utils.u2 1) locals in
      check_reference_local loc;
      let stack = push_local loc stack in
      { locals = locals; stack = stack; }, (operation_of_reference (Utils.u2 1) loc)
  | Instruction.ALOAD_2 ->
      let loc = load (Utils.u2 2) locals in
      check_reference_local loc;
      let stack = push_local loc stack in
      { locals = locals; stack = stack; }, (operation_of_reference (Utils.u2 2) loc)
  | Instruction.ALOAD_3 ->
      let loc = load (Utils.u2 3) locals in
      check_reference_local loc;
      let stack = push_local loc stack in
      { locals = locals; stack = stack; }, (operation_of_reference (Utils.u2 3) loc)
  | Instruction.ANEWARRAY parameter ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = push (value_of_array_element parameter) stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ARETURN ->
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ARRAYLENGTH ->
      let topv = top stack in
      check_reference topv;
      let stack = pop stack in
      let stack = push Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ASTORE parameter ->
      let loc = top stack in
      let stack = pop stack in
      check_reference loc;
      let locals = store (Utils.u2_of_u1 parameter) loc locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ASTORE_0 ->
      let loc = top stack in
      let stack = pop stack in
      check_reference loc;
      let locals = store (Utils.u2 0) loc locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ASTORE_1 ->
      let loc = top stack in
      let stack = pop stack in
      check_reference loc;
      let locals = store (Utils.u2 1) loc locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ASTORE_2 ->
      let loc = top stack in
      let stack = pop stack in
      check_reference loc;
      let locals = store (Utils.u2 2) loc locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ASTORE_3 ->
      let loc = top stack in
      let stack = pop stack in
      check_reference loc;
      let locals = store (Utils.u2 3) loc locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ATHROW ->
      let exc = top stack in
      check_reference exc;
      let stack = empty () in
      let stack = push exc stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.BALOAD ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop stack in
      let stack = push Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.BASTORE ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.BIPUSH x ->
      let x = Int32.of_int (x :> int) in
      let stack = push (Known_integer_value x) stack in
      { locals = locals; stack = stack; }, Push_integer (x, None)
  | Instruction.CALOAD ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop stack in
      let stack = push Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.CASTORE ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.CHECKCAST parameter ->
      let stack = pop stack in
      let stack = push (value_of_parameter_descriptor (match parameter with `Array_type at -> (at :> Descriptor.for_parameter) | `Class_or_interface cn -> `Class cn)) stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.D2F ->
      let topv = top stack in
      let stack = pop_if Unknown_double_value stack in
      (match topv with
      | Unknown_double_value ->
          let stack = push Unknown_float_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_double_value x ->
          let stack = push (Known_float_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Push_float (x, None)]
      | _ -> assert false)
  | Instruction.D2I ->
      let topv = top stack in
      let stack = pop_if Unknown_double_value stack in
      (match topv with
      | Unknown_double_value ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_double_value x ->
          let x = Int32.of_float x in
          let stack = push (Known_integer_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Push_integer (x, None)]
      | _ -> assert false)
  | Instruction.D2L ->
      let topv = top stack in
      let stack = pop_if Unknown_double_value stack in
      (match topv with
      | Unknown_double_value ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_double_value x ->
          let x = Int64.of_float x in
          let stack = push (Known_long_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Push_long (x, None)]
      | _ -> assert false)
  | Instruction.DADD ->
      (* do not perform arithmetical operations on floating-point values *)
      let stack = pop_if Unknown_double_value stack in
      let stack = pop_if Unknown_double_value stack in
      let stack = push Unknown_double_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DALOAD ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop stack in
      let stack = push Unknown_double_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DASTORE ->
      let stack = pop_if Unknown_double_value stack in
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DCMPG ->
      (* do not perform comparisons on floating-point values *)
      let stack = pop_if Unknown_double_value stack in
      let stack = pop_if Unknown_double_value stack in
      let stack = push Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DCMPL ->
      (* do not perform comparisons on floating-point values *)
      let stack = pop_if Unknown_double_value stack in
      let stack = pop_if Unknown_double_value stack in
      let stack = push Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DCONST_0 ->
      let stack = push (Known_double_value 0.0) stack in
      { locals = locals; stack = stack; }, Push_double (0.0, None)
  | Instruction.DCONST_1 ->
      let stack = push (Known_double_value 1.0) stack in
      { locals = locals; stack = stack; }, Push_double (1.0, None)
  | Instruction.DDIV ->
      (* do not perform arithmetical operations on floating-point values *)
      let stack = pop_if Unknown_double_value stack in
      let stack = pop_if Unknown_double_value stack in
      let stack = push Unknown_double_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DLOAD parameter ->
      let loc = check_load (Utils.u2_of_u1 parameter) locals Unknown_double_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_double (Utils.u2_of_u1 parameter) loc)
  | Instruction.DLOAD_0 ->
      let loc = check_load (Utils.u2 0) locals Unknown_double_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_double (Utils.u2 0) loc)
  | Instruction.DLOAD_1 ->
      let loc = check_load (Utils.u2 1) locals Unknown_double_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_double (Utils.u2 1) loc)
  | Instruction.DLOAD_2 ->
      let loc = check_load (Utils.u2 2) locals Unknown_double_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_double (Utils.u2 2) loc)
  | Instruction.DLOAD_3 ->
      let loc = check_load (Utils.u2 3) locals Unknown_double_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_double (Utils.u2 3) loc)
  | Instruction.DMUL ->
      (* do not perform arithmetical operations on floating-point values *)
      let stack = pop_if Unknown_double_value stack in
      let stack = pop_if Unknown_double_value stack in
      let stack = push Unknown_double_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DNEG ->
      (* do not perform arithmetical operations on floating-point values *)
      let stack = pop_if Unknown_double_value stack in
      let stack = push Unknown_double_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DREM ->
      (* do not perform arithmetical operations on floating-point values *)
      let stack = pop_if Unknown_double_value stack in
      let stack = pop_if Unknown_double_value stack in
      let stack = push Unknown_double_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DRETURN ->
      let stack = pop_if Unknown_double_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DSTORE parameter ->
      let topv = top stack in
      let stack = pop_if Unknown_double_value stack in
      let locals = store (Utils.u2_of_u1 parameter) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DSTORE_0 ->
      let topv = top stack in
      let stack = pop_if Unknown_double_value stack in
      let locals = store (Utils.u2 0) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DSTORE_1 ->
      let topv = top stack in
      let stack = pop_if Unknown_double_value stack in
      let locals = store (Utils.u2 1) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DSTORE_2 ->
      let topv = top stack in
      let stack = pop_if Unknown_double_value stack in
      let locals = store (Utils.u2 2) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DSTORE_3 ->
      let topv = top stack in
      let stack = pop_if Unknown_double_value stack in
      let locals = store (Utils.u2 3) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DSUB ->
      (* do not perform arithmetical operations on floating-point values *)
      let stack = pop_if Unknown_double_value stack in
      let stack = pop_if Unknown_double_value stack in
      let stack = push Unknown_double_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DUP ->
      let v, stack = pop_if_category1 stack in
      let stack = push v stack in
      let stack = push v stack in
      { locals = locals; stack = stack; }, operation_of_value v
  | Instruction.DUP2 ->
      let v1 = top stack in
      let stack = pop stack in
      let stack, op =
        if is_category1 v1 then
          let v2, stack = pop_if_category1 stack in
          let stack = push v2 stack in
          let stack = push v1 stack in
          let stack = push v2 stack in
          push v1 stack, Sequence [operation_of_value v2; operation_of_value v1]
        else
          push v1 (push v1 stack), operation_of_value v1 in
      { locals = locals; stack = stack; }, op
  | Instruction.DUP2_X1 ->
      let v1 = top stack in
      let stack =
        if is_category1 v1 then
          let stack = pop stack in
          let v2, stack = pop_if_category1 stack in
          let v3, stack = pop_if_category1 stack in
          let stack = push v2 stack in
          let stack = push v1 stack in
          let stack = push v3 stack in
          let stack = push v2 stack in
          push v1 stack
        else
          let stack = pop stack in
          let v2, stack = pop_if_category1 stack in
          let stack = push v1 stack in
          let stack = push v2 stack in
          push v1 stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DUP2_X2 ->
      let v1 = top stack in
      let stack =
        if is_category1 v1 then begin
          let stack = pop stack in
          let v2, stack = pop_if_category1 stack in
          let v3 = top stack in
          if is_category1 v3 then begin
            let stack = pop stack in
            let v4 = top stack in
            let stack = pop stack in
            let stack = push v2 stack in
            let stack = push v1 stack in
            let stack = push v4 stack in
            let stack = push v3 stack in
            let stack = push v2 stack in
            push v1 stack
          end else begin
            let stack = pop stack in
            let stack = push v2 stack in
            let stack = push v1 stack in
            let stack = push v3 stack in
            let stack = push v2 stack in
            push v1 stack
          end
        end else begin
          let stack = pop stack in
          let v2 = top stack in
          if is_category1 v2 then begin
            let stack = pop stack in
            let v3, stack = pop_if_category1 stack in
            let stack = push v1 stack in
            let stack = push v3 stack in
            let stack = push v2 stack in
            push v1 stack
          end else begin
            let stack = pop stack in
            let stack = push v1 stack in
            let stack = push v2 stack in
            push v1 stack
          end
        end in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DUP_X1 ->
      let v1, stack = pop_if_category1 stack in
      let v2, stack = pop_if_category1 stack in
      let stack = push v1 stack in
      let stack = push v2 stack in
      let stack = push v1 stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.DUP_X2 ->
      let v1, stack = pop_if_category1 stack in
      let v2 = top stack in
      let stack =
        if is_category1 v2 then
          let v2, stack = pop_if_category1 stack in
          let v3, stack = pop_if_category1 stack in
          let stack = push v1 stack in
          let stack = push v3 stack in
          let stack = push v2 stack in
          push v1 stack
        else
          let v2, stack = pop_if_cat2 stack in
          let stack = push v1 stack in
          let stack = push v2 stack in
          push v1 stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.F2D ->
      let topv = top stack in
      let stack = pop_if Unknown_float_value stack in
      (match topv with
      | Unknown_float_value ->
          let stack = push Unknown_double_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_float_value x ->
          let stack = push (Known_double_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Push_double (x, None)]
      | _ -> assert false)
  | Instruction.F2I ->
      let topv = top stack in
      let stack = pop_if Unknown_float_value stack in
      (match topv with
      | Unknown_float_value ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_float_value x ->
          let x = Int32.of_float x in
          let stack = push (Known_integer_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Push_integer (x, None)]
      | _ -> assert false)
  | Instruction.F2L ->
      let topv = top stack in
      let stack = pop_if Unknown_float_value stack in
      (match topv with
      | Unknown_float_value ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_float_value x ->
          let x = Int64.of_float x in
          let stack = push (Known_long_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Push_long (x, None)]
      | _ -> assert false)
  | Instruction.FADD ->
      (* do not perform arithmetical operations on floating-point values *)
      let stack = pop_if Unknown_float_value stack in
      let stack = pop_if Unknown_float_value stack in
      let stack = push Unknown_float_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FALOAD ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop stack in
      let stack = push Unknown_float_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FASTORE ->
      let stack = pop_if Unknown_float_value stack in
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FCMPG ->
      (* do not perform comparisons on floating-point values *)
      let stack = pop_if Unknown_float_value stack in
      let stack = pop_if Unknown_float_value stack in
      let stack = push Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FCMPL ->
      let stack = pop_if Unknown_float_value stack in
      let stack = pop_if Unknown_float_value stack in
      let stack = push Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FCONST_0 ->
      let stack = push (Known_float_value 0.0) stack in
      { locals = locals; stack = stack; }, Push_float (0.0, None)
  | Instruction.FCONST_1 ->
      let stack = push (Known_float_value 1.0) stack in
      { locals = locals; stack = stack; }, Push_float (1.0, None)
  | Instruction.FCONST_2 ->
      let stack = push (Known_float_value 2.0) stack in
      { locals = locals; stack = stack; }, Push_float (2.0, None)
  | Instruction.FDIV ->
      (* do not perform arithmetical operations on floating-point values *)
      let stack = pop_if Unknown_float_value stack in
      let stack = pop_if Unknown_float_value stack in
      let stack = push Unknown_float_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FLOAD parameter ->
      let loc = check_load (Utils.u2_of_u1 parameter) locals Unknown_float_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_float (Utils.u2_of_u1 parameter) loc)
  | Instruction.FLOAD_0 ->
      let loc = check_load (Utils.u2 0) locals Unknown_float_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_float (Utils.u2 0) loc)
  | Instruction.FLOAD_1 ->
      let loc = check_load (Utils.u2 1) locals Unknown_float_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_float (Utils.u2 1) loc)
  | Instruction.FLOAD_2 ->
      let loc = check_load (Utils.u2 2) locals Unknown_float_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_float (Utils.u2 2) loc)
  | Instruction.FLOAD_3 ->
      let loc = check_load (Utils.u2 3) locals Unknown_float_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_float (Utils.u2 3) loc)
  | Instruction.FMUL ->
      (* do not perform arithmetical operations on floating-point values *)
      let stack = pop_if Unknown_float_value stack in
      let stack = pop_if Unknown_float_value stack in
      let stack = push Unknown_float_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FNEG ->
      (* do not perform arithmetical operations on floating-point values *)
      let stack = pop_if Unknown_float_value stack in
      let stack = push Unknown_float_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FREM ->
      (* do not perform arithmetical operations on floating-point values *)
      let stack = pop_if Unknown_float_value stack in
      let stack = pop_if Unknown_float_value stack in
      let stack = push Unknown_float_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FRETURN ->
      let stack = pop_if Unknown_float_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FSTORE parameter ->
      let topv = top stack in
      let stack = pop_if Unknown_float_value stack in
      let locals = store (Utils.u2_of_u1 parameter) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FSTORE_0 ->
      let topv = top stack in
      let stack = pop_if Unknown_float_value stack in
      let locals = store (Utils.u2 0) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FSTORE_1 ->
      let topv = top stack in
      let stack = pop_if Unknown_float_value stack in
      let locals = store (Utils.u2 1) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FSTORE_2 ->
      let topv = top stack in
      let stack = pop_if Unknown_float_value stack in
      let locals = store (Utils.u2 2) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FSTORE_3 ->
      let topv = top stack in
      let stack = pop_if Unknown_float_value stack in
      let locals = store (Utils.u2 3) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.FSUB ->
      (* do not perform arithmetical operations on floating-point values *)
      let stack = pop_if Unknown_float_value stack in
      let stack = pop_if Unknown_float_value stack in
      let stack = push Unknown_float_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.GETFIELD (_, _, desc) ->
      let topv = top stack in
      check_reference topv;
      let stack = pop stack in
      let stack = push (value_of_parameter_descriptor desc) stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.GETSTATIC (_, _, desc) ->
      let stack = push (value_of_parameter_descriptor desc) stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.GOTO _ ->
      { locals = locals; stack = stack; }, Execute i
  | Instruction.GOTO_W _ ->
      { locals = locals; stack = stack; }, Execute i
  | Instruction.I2B ->
      let topv = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match topv with
      | Unknown_integer_value ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_integer_value x ->
          let k = 32 - 8 in
          let x = Int32.shift_right (Int32.shift_left x k) k in
          let stack = push (Known_integer_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Push_integer (x, None)]
      | _ -> assert false)
  | Instruction.I2C ->
      let topv = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match topv with
      | Unknown_integer_value ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_integer_value x ->
          let x = Int32.logand x 0x0000FFFFl in
          let stack = push (Known_integer_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Push_integer (x, None)]
      | _ -> assert false)
  | Instruction.I2D ->
      let topv = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match topv with
      | Unknown_integer_value ->
          let stack = push Unknown_double_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_integer_value x ->
          let x = Int32.to_float x in
          let stack = push (Known_double_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Push_double (x, None)]
      | _ -> assert false)
  | Instruction.I2F ->
      let topv = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match topv with
      | Unknown_integer_value ->
          let stack = push Unknown_float_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_integer_value x ->
          let x = Int32.to_float x in
          let stack = push (Known_float_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Push_float (x, None)]
      | _ -> assert false)
  | Instruction.I2L ->
      let topv = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match topv with
      | Unknown_integer_value ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_integer_value x ->
          let x = Int64.of_int32 x in
          let stack = push (Known_long_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Push_long (x, None)]
      | _ -> assert false)
  | Instruction.I2S ->
      let topv = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match topv with
      | Unknown_integer_value ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_integer_value x ->
          let k = 32 - 16 in
          let x = Int32.shift_right (Int32.shift_left x k) k in
          let stack = push (Known_integer_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Push_integer (x, None)]
      | _ -> assert false)
  | Instruction.IADD ->
      let v2 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match v1, v2 with
      | Known_integer_value x, Known_integer_value y ->
          let res = Int32.add x y in
          let stack = push (Known_integer_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (res, None)]
      | _, Known_integer_value 0l ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Pop
      | _ ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.IALOAD ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop stack in
      let stack = push Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IAND ->
      let v2 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match v1, v2 with
      | Known_integer_value x, Known_integer_value y ->
          let res = Int32.logand x y in
          let stack = push (Known_integer_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (res, None)]
      | _, Known_integer_value (-1l) ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Pop
      | Known_integer_value 0l, _ | _, Known_integer_value 0l ->
          let stack = push (Known_integer_value 0l) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (0l, None)]
      | _ ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.IASTORE ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ICONST_0 ->
      let stack = push (Known_integer_value 0l) stack in
      { locals = locals; stack = stack; }, Push_integer (0l, None)
  | Instruction.ICONST_1 ->
      let stack = push (Known_integer_value 1l) stack in
      { locals = locals; stack = stack; }, Push_integer (1l, None)
  | Instruction.ICONST_2 ->
      let stack = push (Known_integer_value 2l) stack in
      { locals = locals; stack = stack; }, Push_integer (2l, None)
  | Instruction.ICONST_3 ->
      let stack = push (Known_integer_value 3l) stack in
      { locals = locals; stack = stack; }, Push_integer (3l, None)
  | Instruction.ICONST_4 ->
      let stack = push (Known_integer_value 4l) stack in
      { locals = locals; stack = stack; }, Push_integer (4l, None)
  | Instruction.ICONST_5 ->
      let stack = push (Known_integer_value 5l) stack in
      { locals = locals; stack = stack; }, Push_integer (5l, None)
  | Instruction.ICONST_M1 ->
      let stack = push (Known_integer_value (-1l)) stack in
      { locals = locals; stack = stack; }, Push_integer (-1l, None)
  | Instruction.IDIV ->
      let v2 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match v1, v2 with
      | Known_integer_value x, Known_integer_value y when y <> 0l ->
          let res = Int32.div x y in
          let stack = push (Known_integer_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (res, None)]
      | _, Known_integer_value 1l ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Pop
      | _ ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.IF_ACMPEQ _ ->
      let stack = pop stack in
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IF_ACMPNE _ ->
      let stack = pop stack in
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IF_ICMPEQ _ ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IF_ICMPGE _ ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IF_ICMPGT _ ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IF_ICMPLE _ ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IF_ICMPLT _ ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IF_ICMPNE _ ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IFEQ _ ->
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IFGE _ ->
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IFGT _ ->
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IFLE _ ->
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IFLT _ ->
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IFNE _ ->
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IFNONNULL _ ->
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IFNULL _ ->
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IINC (parameter, inc) ->
      let loc = check_load (Utils.u2_of_u1 parameter) locals Unknown_integer_value in
      (match loc with
      | Known_integer_value x ->
          let inc = Int32.of_int (inc :> int) in
          let res = Int32.add x inc in
          let locals = store (Utils.u2_of_u1 parameter) (Known_integer_value res) locals in
          { locals = locals; stack = stack; }, Execute i
      | _ ->
          let locals = store (Utils.u2_of_u1 parameter) Unknown_integer_value locals in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.ILOAD parameter ->
      let loc = check_load (Utils.u2_of_u1 parameter) locals Unknown_integer_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_integer (Utils.u2_of_u1 parameter) loc)
  | Instruction.ILOAD_0 ->
      let loc = check_load (Utils.u2 0) locals Unknown_integer_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_integer (Utils.u2 0) loc)
  | Instruction.ILOAD_1 ->
      let loc = check_load (Utils.u2 1) locals Unknown_integer_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_integer (Utils.u2 1) loc)
  | Instruction.ILOAD_2 ->
      let loc = check_load (Utils.u2 2) locals Unknown_integer_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_integer (Utils.u2 2) loc)
  | Instruction.ILOAD_3 ->
      let loc = check_load (Utils.u2 3) locals Unknown_integer_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_integer (Utils.u2 3) loc)
  | Instruction.IMUL ->
      let v2 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match v1, v2 with
      | Known_integer_value x, Known_integer_value y ->
          let res = Int32.mul x y in
          let stack = push (Known_integer_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (res, None)]
      | _, Known_integer_value 1l ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Pop
      | Known_integer_value 0l, _ | _, Known_integer_value 0l ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (0l, None)]
      | _ ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.INEG ->
      let topv = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match topv with
      | Known_integer_value x ->
          let res = Int32.neg x in
          let stack = push (Known_integer_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Push_integer (res, None)]
      | _ ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.INSTANCEOF _ ->
      let stack = pop stack in
      let stack = push Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.INVOKEDYNAMIC (_, _, (params, ret)) ->
      let infos = List.rev_map value_of_parameter_descriptor params in
      let stack = List.fold_left (fun acc elem -> pop_if elem acc) stack infos in
      let topv = top stack in
      check_reference topv;
      let stack = pop stack in
      let stack = push_return_value ret stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.INVOKEINTERFACE (_, _, (params, ret)) ->
      let infos = List.rev_map value_of_parameter_descriptor params in
      let stack = List.fold_left (fun acc elem -> pop_if elem acc) stack infos in
      let topv = top stack in
      check_reference topv;
      let stack = pop stack in
      let stack = push_return_value ret stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.INVOKESPECIAL (_, _, (params, ret)) ->
      let infos = List.rev_map value_of_parameter_descriptor params in
      let stack = List.fold_left (fun acc elem -> pop_if elem acc) stack infos in
      let topv = top stack in
      check_reference topv;
      let stack = pop stack in
      let stack = push_return_value ret stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.INVOKESTATIC (_, _, (params, ret)) ->
      let infos = List.rev_map value_of_parameter_descriptor params in
      let stack = List.fold_left (fun acc elem -> pop_if elem acc) stack infos in
      let stack = push_return_value ret stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.INVOKEVIRTUAL (_, _, (params, ret)) ->
      let infos = List.rev_map value_of_parameter_descriptor params in
      let stack = List.fold_left (fun acc elem -> pop_if elem acc) stack infos in
      let topv = top stack in
      check_reference topv;
      let stack = pop stack in
      let stack = push_return_value ret stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.IOR ->
      let v2 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match v1, v2 with
      | Known_integer_value x, Known_integer_value y ->
          let res = Int32.logor x y in
          let stack = push (Known_integer_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (res, None)]
      | _, Known_integer_value 0l ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Pop
      | Known_integer_value (-1l), _ | _, Known_integer_value (-1l) ->
          let stack = push (Known_integer_value (-1l)) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (-1l, None)]
      | _ ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.IREM ->
      let v2 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match v1, v2 with
      | Known_integer_value x, Known_integer_value y when y <> 0l ->
          let res = Int32.rem x y in
          let stack = push (Known_integer_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (res, None)]
      | _, Known_integer_value 1l ->
          let stack = push (Known_integer_value 0l) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (0l, None)]
      | _ ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.IRETURN ->
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ISHL ->
      let v2 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match v1, v2 with
      | Known_integer_value x, Known_integer_value y ->
          let res = Int32.shift_left x (Int32.to_int y) in
          let stack = push (Known_integer_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (res, None)]
      | _, Known_integer_value 0l ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Pop
      | _, Known_integer_value y when y > 31l ->
          let stack = push (Known_integer_value 0l) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (0l, None)]
      | _ ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.ISHR ->
      let v2 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match v1, v2 with
      | Known_integer_value x, Known_integer_value y ->
          let res = Int32.shift_right x (Int32.to_int y) in
          let stack = push (Known_integer_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (res, None)]
      | _, Known_integer_value 0l ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Pop
      | _ ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.ISTORE parameter ->
      let topv = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let locals = store (Utils.u2_of_u1 parameter) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ISTORE_0 ->
      let topv = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let locals = store (Utils.u2 0) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ISTORE_1 ->
      let topv = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let locals = store (Utils.u2 1) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ISTORE_2 ->
      let topv = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let locals = store (Utils.u2 2) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ISTORE_3 ->
      let topv = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let locals = store (Utils.u2 3) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.ISUB ->
      let v2 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match v1, v2 with
      | Known_integer_value x, Known_integer_value y ->
          let res = Int32.sub x y in
          let stack = push (Known_integer_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (res, None)]
      | _, Known_integer_value 0l ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Pop
      | _ ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.IUSHR ->
      let v2 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match v1, v2 with
      | Known_integer_value x, Known_integer_value y ->
          let res = Int32.shift_right_logical x (Int32.to_int y) in
          let stack = push (Known_integer_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (res, None)]
      | _, Known_integer_value 0l ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Pop
      | _, Known_integer_value y when y > 31l ->
          let stack = push (Known_integer_value 0l) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (0l, None)]
      | _ ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.IXOR ->
      let v2 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      (match v1, v2 with
      | Known_integer_value x, Known_integer_value y ->
          let res = Int32.logxor x y in
          let stack = push (Known_integer_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop; Push_integer (res, None)]
      | _, Known_integer_value 0l ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Pop
      | _ ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.JSR _ ->
      fail (Unsupported_instruction "JSR")
  | Instruction.JSR_W _ ->
      fail (Unsupported_instruction "JSR_W")
  | Instruction.L2D ->
      let topv = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match topv with
      | Unknown_long_value ->
          let stack = push Unknown_double_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_long_value x ->
          let x = Int64.to_float x in
          let stack = push (Known_double_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Push_double (x, None)]
      | _ -> assert false)
  | Instruction.L2F ->
      let topv = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match topv with
      | Unknown_long_value ->
          let stack = push Unknown_float_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_long_value x ->
          let x = Int64.to_float x in
          let stack = push (Known_float_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Push_float (x, None)]
      | _ -> assert false)
  | Instruction.L2I ->
      let topv = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match topv with
      | Unknown_long_value ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i
      | Known_long_value x ->
          let x = Int64.to_int32 x in
          let stack = push (Known_integer_value x) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Push_integer (x, None)]
      | _ -> assert false)
  | Instruction.LADD ->
      let v2 = top stack in
      let stack = pop_if Unknown_long_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match v1, v2 with
      | Known_long_value x, Known_long_value y ->
          let res = Int64.add x y in
          let stack = push (Known_long_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Pop2; Push_long (res, None)]
      | _, Known_long_value 0L ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Pop2
      | _ ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.LALOAD ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop stack in
      let stack = push Unknown_long_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.LAND ->
      let v2 = top stack in
      let stack = pop_if Unknown_long_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match v1, v2 with
      | Known_long_value x, Known_long_value y ->
          let res = Int64.logand x y in
          let stack = push (Known_long_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Pop2; Push_long (res, None)]
      | _, Known_long_value (-1L) ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Pop2
      | Known_long_value 0L, _ | _, Known_long_value 0L ->
          let stack = push (Known_long_value 0L) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Pop2; Push_long (0L, None)]
      | _ ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.LASTORE ->
      let stack = pop_if Unknown_long_value stack in
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.LCMP ->
      let v2 = top stack in
      let stack = pop_if Unknown_long_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match v1, v2 with
      | Known_long_value x, Known_long_value y ->
          let res =
            if x > y then
              1l
            else if x < y then
              -1l
            else
              0l in
          let stack = push (Known_integer_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Pop2; Push_integer (res, None)]
      | _ ->
          let stack = push Unknown_integer_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.LCONST_0 ->
      let stack = push (Known_long_value 0L) stack in
      { locals = locals; stack = stack; }, Push_long (0L, None)
  | Instruction.LCONST_1 ->
      let stack = push (Known_long_value 1L) stack in
      { locals = locals; stack = stack; }, Push_long (1L, None)
  | Instruction.LDC parameter ->
      let stack = push (value_of_constant_descriptor parameter) stack in
      { locals = locals; stack = stack; }, (operation_of_constant_descriptor parameter)
  | Instruction.LDC2_W parameter ->
      let stack = push (value_of_constant_descriptor parameter) stack in
      { locals = locals; stack = stack; }, (operation_of_constant_descriptor parameter)
  | Instruction.LDC_W parameter ->
      let stack = push (value_of_constant_descriptor parameter) stack in
      { locals = locals; stack = stack; }, (operation_of_constant_descriptor parameter)
  | Instruction.LDIV ->
      let v2 = top stack in
      let stack = pop_if Unknown_long_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match v1, v2 with
      | Known_long_value x, Known_long_value y when y <> 0L ->
          let res = Int64.div x y in
          let stack = push (Known_long_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Pop2; Push_long (res, None)]
      | _, Known_long_value 1L ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Pop2
      | _ ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.LLOAD parameter ->
      let loc = check_load (Utils.u2_of_u1 parameter) locals Unknown_long_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_long (Utils.u2_of_u1 parameter) loc)
  | Instruction.LLOAD_0 ->
      let loc = check_load (Utils.u2 0) locals Unknown_long_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_long (Utils.u2 0) loc)
  | Instruction.LLOAD_1 ->
      let loc = check_load (Utils.u2 1) locals Unknown_long_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_long (Utils.u2 1) loc)
  | Instruction.LLOAD_2 ->
      let loc = check_load (Utils.u2 2) locals Unknown_long_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_long (Utils.u2 2) loc)
  | Instruction.LLOAD_3 ->
      let loc = check_load (Utils.u2 3) locals Unknown_long_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_long (Utils.u2 3) loc)
  | Instruction.LMUL ->
      let v2 = top stack in
      let stack = pop_if Unknown_long_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match v1, v2 with
      | Known_long_value x, Known_long_value y ->
          let res = Int64.mul x y in
          let stack = push (Known_long_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Pop2; Push_long (res, None)]
      | _, Known_long_value 1L ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Pop2
      | Known_long_value 0L, _ | _, Known_long_value 0L ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Pop2; Push_long (0L, None)]
      | _ ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.LNEG ->
      let topv = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match topv with
      | Known_long_value x ->
          let res = Int64.neg x in
          let stack = push (Known_long_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Push_long (res, None)]
      | _ ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.LOOKUPSWITCH _ ->
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.LOR ->
      let v2 = top stack in
      let stack = pop_if Unknown_long_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match v1, v2 with
      | Known_long_value x, Known_long_value y ->
          let res = Int64.logor x y in
          let stack = push (Known_long_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Pop2; Push_long (res, None)]
      | _, Known_long_value 0L ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Pop2
      | Known_long_value (-1L), _ | _, Known_long_value (-1L) ->
          let stack = push (Known_long_value (-1L)) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Pop2; Push_long (-1L, None)]
      | _ ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.LREM ->
      let v2 = top stack in
      let stack = pop_if Unknown_long_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match v1, v2 with
      | Known_long_value x, Known_long_value y when y <> 0L->
          let res = Int64.rem x y in
          let stack = push (Known_long_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Pop2; Push_long (res, None)]
      | _, Known_long_value 1L ->
          let stack = push (Known_long_value 0L) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Pop2; Push_long (0L, None)]
      | _ ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.LRETURN ->
      let stack = pop_if Unknown_long_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.LSHL ->
      let v2 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match v1, v2 with
      | Known_long_value x, Known_integer_value y ->
          let res = Int64.shift_left x (Int32.to_int y) in
          let stack = push (Known_long_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop2; Push_long (res, None)]
      | _, Known_integer_value 0l ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Pop2
      | _, Known_integer_value y when y > 63l ->
          let stack = push (Known_long_value 0L) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop2; Push_long (0L, None)]
      | _ ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.LSHR ->
      let v2 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match v1, v2 with
      | Known_long_value x, Known_integer_value y ->
          let res = Int64.shift_right x (Int32.to_int y) in
          let stack = push (Known_long_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop2; Push_long (res, None)]
      | _, Known_integer_value 0l ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Pop2
      | _ ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.LSTORE parameter ->
      let topv = top stack in
      let stack = pop_if Unknown_long_value stack in
      let locals = store (Utils.u2_of_u1 parameter) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.LSTORE_0 ->
      let topv = top stack in
      let stack = pop_if Unknown_long_value stack in
      let locals = store (Utils.u2 0) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.LSTORE_1 ->
      let topv = top stack in
      let stack = pop_if Unknown_long_value stack in
      let locals = store (Utils.u2 1) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.LSTORE_2 ->
      let topv = top stack in
      let stack = pop_if Unknown_long_value stack in
      let locals = store (Utils.u2 2) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.LSTORE_3 ->
      let topv = top stack in
      let stack = pop_if Unknown_long_value stack in
      let locals = store (Utils.u2 3) topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.LSUB ->
      let v2 = top stack in
      let stack = pop_if Unknown_long_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match v1, v2 with
      | Known_long_value x, Known_long_value y ->
          let res = Int64.sub x y in
          let stack = push (Known_long_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Pop2; Push_long (res, None)]
      | _, Known_long_value 0L ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Pop2
      | _ ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.LUSHR ->
      let v2 = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match v1, v2 with
      | Known_long_value x, Known_integer_value y ->
          let res = Int64.shift_right_logical x (Int32.to_int y) in
          let stack = push (Known_long_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop2; Push_long (res, None)]
      | _, Known_integer_value 0l ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Pop2
      | _, Known_integer_value y when y > 63l ->
          let stack = push (Known_long_value 0L) stack in
          { locals = locals; stack = stack; }, Sequence [Pop; Pop2; Push_long (0L, None)]
      | _ ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.LXOR ->
      let v2 = top stack in
      let stack = pop_if Unknown_long_value stack in
      let v1 = top stack in
      let stack = pop_if Unknown_long_value stack in
      (match v1, v2 with
      | Known_long_value x, Known_long_value y ->
          let res = Int64.logxor x y in
          let stack = push (Known_long_value res) stack in
          { locals = locals; stack = stack; }, Sequence [Pop2; Pop2; Push_long (res, None)]
      | _, Known_long_value 0L ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Pop2
      | _ ->
          let stack = push Unknown_long_value stack in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.MONITORENTER ->
      let topv = top stack in
      check_reference topv;
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.MONITOREXIT ->
      let topv = top stack in
      check_reference topv;
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.MULTIANEWARRAY (_, dims) ->
      let s = ref stack in
      for _i = 1 to (dims :> int) do
        s := pop_if Unknown_integer_value !s;
      done;
      let stack = push Object_value !s in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.NEW _ ->
      let stack = push Object_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.NEWARRAY parameter ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = push (value_of_array_primitive parameter) stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.NOP ->
      { locals = locals; stack = stack; }, Execute i
  | Instruction.POP ->
      let _, stack = pop_if_category1 stack in
      { locals = locals; stack = stack; }, Pop
  | Instruction.POP2 ->
      let v1 = top stack in
      let stack =
        if is_category1 v1 then
          snd (pop_if_category1 (snd (pop_if_category1 stack)))
        else
          pop stack in
      { locals = locals; stack = stack; }, Pop2
  | Instruction.PUTFIELD (_, _, desc) ->
      let stack = pop_if (value_of_parameter_descriptor desc) stack in
      let topv = top stack in
      check_reference topv;
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.PUTSTATIC (_, _, desc) ->
      let stack = pop_if (value_of_parameter_descriptor desc) stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.RET _ ->
      fail (Unsupported_instruction "RET")
  | Instruction.RETURN ->
      { locals = locals; stack = stack; }, Execute i
  | Instruction.SALOAD ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop stack in
      let stack = push Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.SASTORE ->
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop_if Unknown_integer_value stack in
      let stack = pop stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.SIPUSH x ->
      let x = Int32.of_int (x :> int) in
      let stack = push (Known_integer_value x) stack in
      { locals = locals; stack = stack; }, Push_integer (x, None)
  | Instruction.SWAP ->
      let v1, stack = pop_if_category1 stack in
      let v2, stack = pop_if_category1 stack in
      let stack = push v1 stack in
      let stack = push v2 stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.TABLESWITCH _ ->
      let stack = pop_if Unknown_integer_value stack in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.WIDE_ALOAD parameter ->
      let loc = load parameter locals in
      check_reference_local loc;
      let stack = push_local loc stack in
      { locals = locals; stack = stack; }, (operation_of_reference parameter loc)
  | Instruction.WIDE_ASTORE parameter ->
      let loc = top stack in
      let stack = pop stack in
      check_reference loc;
      let locals = store parameter loc locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.WIDE_DLOAD parameter ->
      let loc = check_load parameter locals Unknown_double_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_double parameter loc)
  | Instruction.WIDE_DSTORE parameter ->
      let topv = top stack in
      let stack = pop_if Unknown_double_value stack in
      let locals = store parameter topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.WIDE_FLOAD parameter ->
      let loc = check_load parameter locals Unknown_float_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_float parameter loc)
  | Instruction.WIDE_FSTORE parameter ->
      let topv = top stack in
      let stack = pop_if Unknown_float_value stack in
      let locals = store parameter topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.WIDE_IINC (parameter, inc) ->
      let loc = check_load parameter locals Unknown_integer_value in
      (match loc with
      | Known_integer_value x ->
          let inc = Int32.of_int (inc :> int) in
          let res = Int32.add x inc in
          let locals = store parameter (Known_integer_value res) locals in
          { locals = locals; stack = stack; }, Execute i
      | _ ->
          let locals = store parameter Unknown_integer_value locals in
          { locals = locals; stack = stack; }, Execute i)
  | Instruction.WIDE_ILOAD parameter ->
      let loc = check_load parameter locals Unknown_integer_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_integer parameter loc)
  | Instruction.WIDE_ISTORE parameter ->
      let topv = top stack in
      let stack = pop_if Unknown_integer_value stack in
      let locals = store parameter topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.WIDE_LLOAD parameter ->
      let loc = check_load parameter locals Unknown_long_value in
      let stack = push loc stack in
      { locals = locals; stack = stack; }, (operation_of_long parameter loc)
  | Instruction.WIDE_LSTORE parameter ->
      let topv = top stack in
      let stack = pop_if Unknown_long_value stack in
      let locals = store parameter topv locals in
      { locals = locals; stack = stack; }, Execute i
  | Instruction.WIDE_RET _ ->
      fail (Unsupported_instruction "WIDE_RET")

let is_long = function
  | Unknown_long_value
  | Known_long_value _ -> true
  | _ -> false

let is_double = function
  | Unknown_double_value
  | Known_double_value _ -> true
  | _ -> false

let unify ev1 ev2 =
  let unify_values v1 v2 =
    if v1 == v2 then
      v1
    else
      match v1, v2 with
      (* Unknown_value is absorbing *)
      | Unknown_value, _ | _, Unknown_value -> Unknown_value
      (* integer values *)
      | Unknown_integer_value, Unknown_integer_value -> Unknown_integer_value
      | Unknown_integer_value, Known_integer_value _ -> Unknown_integer_value
      | Known_integer_value _, Unknown_integer_value -> Unknown_integer_value
      | Known_integer_value x, Known_integer_value y ->
          if x = y then Known_integer_value x else Unknown_integer_value
      (* float values *)
      | Unknown_float_value, Unknown_float_value -> Unknown_float_value
      | Unknown_float_value, Known_float_value _ -> Unknown_float_value
      | Known_float_value _, Unknown_float_value -> Unknown_float_value
      | Known_float_value x, Known_float_value y ->
          if x = y then Known_float_value x else Unknown_float_value
      (* long values *)
      | Unknown_long_value, Unknown_long_value -> Unknown_long_value
      | Unknown_long_value, Known_long_value _ -> Unknown_long_value
      | Known_long_value _, Unknown_long_value -> Unknown_long_value
      | Known_long_value x, Known_long_value y ->
          if x = y then Known_long_value x else Unknown_long_value
      (* double values *)
      | Unknown_double_value, Unknown_double_value -> Unknown_double_value
      | Unknown_double_value, Known_double_value _ -> Unknown_double_value
      | Known_double_value _, Unknown_double_value -> Unknown_double_value
      | Known_double_value x, Known_double_value y ->
          if x = y then Known_double_value x else Unknown_double_value
      (* null and objects *)
      | Null_value, Object_value -> v2
      | Object_value, Null_value -> v1
      | _ -> if v1 = v2 then v1 else Unknown_value in
  let sz1 = List.length ev1.stack in
  let sz2 = List.length ev2.stack in
  if sz1 = sz2 then begin
    let len1 = Array.length ev1.locals in
    let len2 = Array.length ev2.locals in
    let len = Utils.max_int len1 len2 in
    let locals1 =
      Array.init
        len
        (fun i -> if i < len1 then ev1.locals.(i) else Local Unknown_value) in
    let locals2 =
      Array.init
        len
        (fun i -> if i < len2 then ev2.locals.(i) else Local Unknown_value) in
    let locals = Array.make len Place_holder in
    let i = ref 0 in
    while !i < len do
      match locals1.(!i), locals2.(!i) with
      | Local l1, Local l2 when (is_long l1) && (is_long l2) ->
          locals.(!i) <- Local (unify_values l1 l2);
          incr i;
          locals.(!i) <- Place_holder;
          incr i
      | Local d1, Local d2 when (is_double d1) && (is_double d2) ->
          locals.(!i) <- Local (unify_values d1 d2);
          incr i;
          locals.(!i) <- Place_holder;
          incr i
      | Local Unknown_long_value, Local _
      | Local (Known_long_value _), Local _
      | Local _, Local Unknown_long_value
      | Local _, Local (Known_long_value _)
      | Local Unknown_double_value, Local _
      | Local (Known_double_value _), Local _
      | Local _, Local Unknown_double_value
      | Local _, Local (Known_double_value _) ->
          locals.(!i) <- Local Unknown_value;
          incr i;
          locals.(!i) <- Local Unknown_value;
          incr i
      | Local loc1, Local loc2 ->
          locals.(!i) <- Local (unify_values loc1 loc2);
          incr i
      | Place_holder, _
      | _, Place_holder ->
          locals.(!i) <- Local Unknown_value;
          incr i
    done;
    let stack = List.map2 unify_values ev1.stack ev2.stack in
    { locals = locals; stack = stack; }
  end else
    fail (Different_stack_sizes (sz1, sz2))

let unify ev1 ev2 =
  if (ev1 == ev2) then
    ev1
  else
    unify ev1 ev2
