(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


(* Types *)

type local =
  | Local of Attribute.verification_type_info
  | Place_holder_long
  | Place_holder_double

type locals = local array

type stack = Attribute.verification_type_info list (* stack top is list head *)

type t = {
    locals : locals;
    stack : stack;
  }


(* Exception *)

type offset = Utils.u2

BARISTA_ERROR =
  | Unsupported_instruction of (ofs : offset) * (x : string) ->
      Printf.sprintf "unsupported instruction at offset %d: %S" (ofs :> int) x
  | Empty_stack of (ofs : offset) ->
      Printf.sprintf "empty stack at offset %d" (ofs :> int)
  | Invalid_local_index of (ofs : offset) * (i : Utils.u2) * (l : int) ->
      Printf.sprintf "invalid local index at offset %d (%d, length %d)"
        (ofs :> int)
        (i :> int) l
  | Invalid_stack_top of (ofs : offset) * (w : Attribute.verification_type_info) * (f : Attribute.verification_type_info) ->
      Printf.sprintf "invalid stack top at offset %d: %S waited but %S found"
        (ofs :> int)
        (Attribute.string_of_verification_type_info w)
        (Attribute.string_of_verification_type_info f)
  | Invalid_local_contents of (ofs : offset) * (i : Utils.u2) * (w : Attribute.verification_type_info) * (f : Attribute.verification_type_info) ->
      Printf.sprintf "invalid local contents at offset %d for index %d: %S waited but %S found"
        (ofs :> int)
        (i :> int)
        (Attribute.string_of_verification_type_info w)
        (Attribute.string_of_verification_type_info f)
  | Invalid_local_contents_placeholder of (ofs : offset) * (i : Utils.u2) * (w : Attribute.verification_type_info) ->
      Printf.sprintf "invalid local contents at offset %d for index %d: %S waited but placeholder found"
        (ofs :> int)
        (i :> int)
        (Attribute.string_of_verification_type_info w)
  | Reference_waited of (ofs : offset) * (f : Attribute.verification_type_info) ->
      Printf.sprintf "reference waited but %S found at offset %d"
        (Attribute.string_of_verification_type_info f)
        (ofs :> int)
  | Reference_waited_placeholder_found of (ofs : offset) ->
      Printf.sprintf "reference waited but placeholder found at offset %d"
        (ofs :> int)
  | Array_waited of (ofs : offset) ->
      Printf.sprintf "array waited at offset %d"
        (ofs :> int)
  | Category1_waited of (ofs : offset) ->
      Printf.sprintf "category1 waited at offset %d"
        (ofs :> int)
  | Category2_waited of (ofs : offset) ->
      Printf.sprintf "category2 waited at offset %d"
        (ofs :> int)
  | Different_stack_sizes of (sz1 : int) * (sz2 : int) ->
      Printf.sprintf "different stack sizes (%d and %d)"
        sz1
        sz2
  | Invalid_primitive_array_type of (ofs : offset) ->
      Printf.sprintf "invalid primitive array type at offset %d"
        (ofs :> int)
  | Empty_frame_list ->
      "empty frame list"
  | Different_frames of (ofs : offset) ->
      Printf.sprintf "different frames at offset %d"
        (ofs :> int)


(* Construction *)

let make_empty () =
  { locals = [||]; stack = []; }

let java_lang_String = Name.make_for_class_from_external @"java.lang.String"

let java_lang_Class = Name.make_for_class_from_external @"java.lang.Class"

let java_lang_invoke_MethodType = Name.make_for_class_from_external @"java.lang.invoke.MethodType"

let java_lang_invoke_MethodHandle = Name.make_for_class_from_external @"java.lang.invoke.MethodHandle"

let verification_type_info_of_constant_descriptor = function
  | `Int _ -> Attribute.Integer_variable_info
  | `Float _ -> Attribute.Float_variable_info
  | `String _ -> Attribute.Object_variable_info (`Class_or_interface java_lang_String)
  | `Class_or_interface _ -> Attribute.Object_variable_info (`Class_or_interface java_lang_Class)
  | `Array_type _ -> Attribute.Object_variable_info (`Class_or_interface java_lang_Class)
  | `Long _ -> Attribute.Long_variable_info
  | `Double _ -> Attribute.Double_variable_info
  | `Interface_method _ -> Attribute.Object_variable_info (`Class_or_interface java_lang_invoke_MethodHandle)
  | `Method_type _ -> Attribute.Object_variable_info (`Class_or_interface java_lang_invoke_MethodType)
  | `Method_handle _ -> Attribute.Object_variable_info (`Class_or_interface java_lang_invoke_MethodHandle)

let of_list l =
  l
  |> List.map
      (function
        | Attribute.Long_variable_info ->
            [ Local Attribute.Long_variable_info; Place_holder_long ]
        | Attribute.Double_variable_info ->
            [ Local Attribute.Double_variable_info; Place_holder_double ]
        | x -> [ Local x ])
  |> List.concat
  |> Array.of_list

let make_of_parameters c l =
  let l = List.map Attribute.verification_type_info_of_parameter_descriptor l in
  let l = match c with
  | Some (_, true) -> Attribute.Uninitialized_this_variable_info :: l
  | Some (cn, false) -> (Attribute.Object_variable_info (`Class_or_interface cn)) :: l
  | None -> l in
  { locals = of_list l; stack = []; }

let make_of_method cn = function
  | Method.Regular { Method.flags; descriptor; _ } ->
      let l =
        descriptor
        |> fst
        |> List.map Attribute.verification_type_info_of_parameter_descriptor in
      let l =
        if AccessFlag.mem_method `Static flags then
          l
        else
          (Attribute.Object_variable_info (`Class_or_interface cn)) :: l in
      { locals = of_list l; stack = [] }
  | Method.Constructor { Method.cstr_descriptor = l ; _ } ->
      let l = List.map Attribute.verification_type_info_of_parameter_descriptor l in
      let l = Attribute.Uninitialized_this_variable_info :: l in
      { locals = of_list l; stack = [] }
  | Method.Initializer _ ->
      make_empty ()


(* Access and modification *)

let locals_size st =
  Array.length st.locals

let stack_size st =
  List.fold_left
    (fun acc x ->
      acc +
        (match x with
        | Attribute.Double_variable_info
        | Attribute.Long_variable_info -> 2
        | _ -> 1))
    0
    st.stack

let array_for_all2 ~n ~eq a1 a2 =
  let i = ref 0 in
  while (!i < n) && (eq a1.(!i) a2.(!i)) do
    incr i
  done;
  !i = n

let same_local loc1 loc2 =
  match loc1, loc2 with
  | Local l1, Local l2 ->
      Attribute.equal_verification_type_info l1 l2
  | Place_holder_long, Place_holder_long
  | Place_holder_double, Place_holder_double ->
      true
  | _ ->
      false

let same_locals st1 st2 =
  ((locals_size st1) = (locals_size st2))
    && (array_for_all2
          ~n:(locals_size st1)
          ~eq:same_local
          st1.locals
          st2.locals)

let same_stack st1 st2 =
  ((stack_size st1) = (stack_size st2))
    && (List.for_all2 Attribute.equal_verification_type_info st1.stack st2.stack)

let equal st1 st2 =
  (st1 == st2)
|| ((same_locals st1 st2) && (same_stack st1 st2))

let push v s =
  v :: s

let push_local v s =
  match v with
  | Local v -> push v s
  | Place_holder_long -> assert false
  | Place_holder_double -> assert false

let push_return_value x s =
  match x with
  | `Void -> s
  | #Descriptor.for_parameter as y ->
      push (Attribute.verification_type_info_of_parameter_descriptor y) s

let top ofs = function
  | hd :: _ -> hd
  | [] -> fail (Empty_stack ofs)

let pop ofs = function
  | _ :: tl -> tl
  | [] -> fail (Empty_stack ofs)

let pop_if ofs v s =
  let v' = top ofs s in
  let popable = match v with
  | Attribute.Object_variable_info _ -> true
  | _ -> Attribute.equal_verification_type_info v v' in
  if popable then
    pop ofs s
  else
    fail (Invalid_stack_top (ofs, v, v'))

let is_category1 = function
  | Attribute.Top_variable_info
  | Attribute.Integer_variable_info
  | Attribute.Float_variable_info
  | Attribute.Null_variable_info
  | Attribute.Uninitialized_this_variable_info
  | Attribute.Object_variable_info _
  | Attribute.Uninitialized_variable_info _ -> true
  | Attribute.Long_variable_info
  | Attribute.Double_variable_info -> false

let pop_if_category1 ofs = function
  | hd :: tl -> if is_category1 hd then hd, tl else fail (Category1_waited ofs)
  | [] -> fail (Empty_stack ofs)

let pop_if_cat2 ofs = function
  | hd :: tl -> if not (is_category1 hd) then hd, tl else fail (Category2_waited ofs)
  | [] -> fail (Empty_stack ofs)

let empty () =
  []

let only_exception cn =
  [Attribute.Object_variable_info (`Class_or_interface cn)]

let load ofs i l =
  let j = (i : Utils.u2 :> int) in
  let len = Array.length l in
  if j >= 0 && j < len then
    l.(j)
  else
    fail (Invalid_local_index (ofs, i, len))

let check_load ofs i l v =
  let v' = load ofs i l in
  match v' with
  | Local v' ->
      if not (Attribute.equal_verification_type_info v v') then
        fail (Invalid_local_contents (ofs, i, v, v'))
  | Place_holder_long | Place_holder_double ->
      fail (Invalid_local_contents_placeholder (ofs, i, v))

let store i v l =
  let long_or_double = function
    | Attribute.Long_variable_info
    | Attribute.Double_variable_info -> true
    | _ -> false in
  let long_or_double_local = function
    | Local l -> long_or_double l
    | _ -> false in
  let placeholder = function
    | Local _ -> false
    | Place_holder_long -> true
    | Place_holder_double -> true in
  let i = (i : Utils.u2 :> int) in
  let len_l = Array.length l in
  let is_long_or_double = long_or_double v in
  let len_for_store = (succ i) + (if is_long_or_double then 1 else 0) in
  let len_res = Utils.max_int len_l len_for_store in
  let res =
    Array.init
      len_res
      (fun i -> if i < len_l then l.(i) else Local Attribute.Top_variable_info) in
  (* always set the given index *)
  res.(i) <- Local v;
  (* if the store replaces a placeholder, change predecessor to top *)
  if (i > 0) && (i < len_l) && (placeholder l.(i)) then
    res.(pred i) <- Local Attribute.Top_variable_info;
  if is_long_or_double then begin
    (* if the element is a long/double, change successor to placeholder *)
    let j = succ i in
    res.(j) <-
      if v = Attribute.Long_variable_info
      then Place_holder_long
      else Place_holder_double;
    (* if the successor was a long/double, change the following one to top *)
    if (j >= 0) && (j < len_l) && (long_or_double_local l.(j)) then
      res.(succ j) <- Local Attribute.Top_variable_info
  end else begin
    (* if the element is not a long/double but was one, change successor to top *)
    if (i >= 0) && (i < len_l) && (long_or_double_local l.(i)) then
      res.(succ i) <- Local Attribute.Top_variable_info
  end;
  res

(* Operations *)

let check_reference ofs x =
  match x with
  | Attribute.Integer_variable_info
  | Attribute.Float_variable_info
  | Attribute.Long_variable_info
  | Attribute.Double_variable_info
  | Attribute.Top_variable_info -> fail (Reference_waited (ofs, x))
  | Attribute.Null_variable_info
  | Attribute.Uninitialized_this_variable_info
  | Attribute.Object_variable_info _
  | Attribute.Uninitialized_variable_info _ -> ()

let check_reference_local ofs x =
  match x with
  | Local x ->
      check_reference ofs x
  | Place_holder_long | Place_holder_double ->
      fail (Reference_waited_placeholder_found ofs)

let enclose (x : Descriptor.array_type) =
  `Array (x :> Descriptor.for_field)

let verification_type_info_of_array_element = function
  | `Array_type at -> Attribute.Object_variable_info (`Array_type (enclose at))
  | `Class_or_interface cn -> Attribute.Object_variable_info (`Array_type (`Array (`Class cn)))

let verification_type_info_of_array_primitive ofs = function
  | `Boolean -> Attribute.Object_variable_info (`Array_type (`Array `Boolean))
  | `Char -> Attribute.Object_variable_info (`Array_type (`Array `Char))
  | `Float -> Attribute.Object_variable_info (`Array_type (`Array `Float))
  | `Double -> Attribute.Object_variable_info (`Array_type (`Array `Double))
  | `Byte -> Attribute.Object_variable_info (`Array_type (`Array `Byte))
  | `Short -> Attribute.Object_variable_info (`Array_type (`Array `Short))
  | `Int -> Attribute.Object_variable_info (`Array_type (`Array `Int))
  | `Long -> Attribute.Object_variable_info (`Array_type (`Array `Long))
  | _ -> fail (Invalid_primitive_array_type ofs)

let update class_name ofs i st =
  let locals = Array.copy st.locals in
  let stack = st.stack in
  match i with
  | Instruction.AALOAD ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let topv = top ofs stack in
      check_reference ofs topv;
      let stack = pop ofs stack in
      let stack =
        (match topv with
        | Attribute.Null_variable_info -> push Attribute.Null_variable_info stack
        | Attribute.Object_variable_info (`Array_type (`Array t)) -> push (Attribute.verification_type_info_of_parameter_descriptor t) stack
        | _ -> fail (Array_waited ofs)) in
      { locals = locals; stack = stack; }
  | Instruction.AASTORE ->
      let topv = top ofs stack in
      check_reference ofs topv;
      let stack = pop ofs stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let topv = top ofs stack in
      check_reference ofs topv;
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.ACONST_NULL ->
      let stack = push Attribute.Null_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ALOAD parameter ->
      let loc = load ofs (Utils.u2_of_u1 parameter) locals in
      check_reference_local ofs loc;
      let stack = push_local loc stack in
      { locals = locals; stack = stack; }
  | Instruction.ALOAD_0 ->
      let loc = load ofs (Utils.u2 0) locals in
      check_reference_local ofs loc;
      let stack = push_local loc stack in
      { locals = locals; stack = stack; }
  | Instruction.ALOAD_1 ->
      let loc = load ofs (Utils.u2 1) locals in
      check_reference_local ofs loc;
      let stack = push_local loc stack in
      { locals = locals; stack = stack; }
  | Instruction.ALOAD_2 ->
      let loc = load ofs (Utils.u2 2) locals in
      check_reference_local ofs loc;
      let stack = push_local loc stack in
      { locals = locals; stack = stack; }
  | Instruction.ALOAD_3 ->
      let loc = load ofs (Utils.u2 3) locals in
      check_reference_local ofs loc;
      let stack = push_local loc stack in
      { locals = locals; stack = stack; }
  | Instruction.ANEWARRAY parameter ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push (verification_type_info_of_array_element parameter) stack in
      { locals = locals; stack = stack; }
  | Instruction.ARETURN ->
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.ARRAYLENGTH ->
      let topv = top ofs stack in
      check_reference ofs topv;
      let stack = pop ofs stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ASTORE parameter ->
      let loc = top ofs stack in
      let stack = pop ofs stack in
      check_reference ofs loc;
      let locals = store (Utils.u2_of_u1 parameter) loc locals in
      { locals = locals; stack = stack; }
  | Instruction.ASTORE_0 ->
      let loc = top ofs stack in
      let stack = pop ofs stack in
      check_reference ofs loc;
      let locals = store (Utils.u2 0) loc locals in
      { locals = locals; stack = stack; }
  | Instruction.ASTORE_1 ->
      let loc = top ofs stack in
      let stack = pop ofs stack in
      check_reference ofs loc;
      let locals = store (Utils.u2 1) loc locals in
      { locals = locals; stack = stack; }
  | Instruction.ASTORE_2 ->
      let loc = top ofs stack in
      let stack = pop ofs stack in
      check_reference ofs loc;
      let locals = store (Utils.u2 2) loc locals in
      { locals = locals; stack = stack; }
  | Instruction.ASTORE_3 ->
      let loc = top ofs stack in
      let stack = pop ofs stack in
      check_reference ofs loc;
      let locals = store (Utils.u2 3) loc locals in
      { locals = locals; stack = stack; }
  | Instruction.ATHROW ->
      let exc = top ofs stack in
      check_reference ofs exc;
      let stack = empty () in
      let stack = push exc stack in
      { locals = locals; stack = stack; }
  | Instruction.BALOAD ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop ofs stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.BASTORE ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.BIPUSH _ ->
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.CALOAD ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop ofs stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.CASTORE ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.CHECKCAST parameter ->
      let stack = pop ofs stack in
      let stack = push (Attribute.verification_type_info_of_parameter_descriptor (match parameter with `Array_type at -> (at :> Descriptor.for_parameter) | `Class_or_interface cn -> `Class cn)) stack in
      { locals = locals; stack = stack; }
  | Instruction.D2F ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.D2I ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.D2L ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DADD ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DALOAD ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop ofs stack in
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DASTORE ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.DCMPG ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DCMPL ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DCONST_0 ->
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DCONST_1 ->
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DDIV ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DLOAD parameter ->
      check_load ofs (Utils.u2_of_u1 parameter) locals Attribute.Double_variable_info;
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DLOAD_0 ->
      check_load ofs (Utils.u2 0) locals Attribute.Double_variable_info;
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DLOAD_1 ->
      check_load ofs (Utils.u2 1) locals Attribute.Double_variable_info;
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DLOAD_2 ->
      check_load ofs (Utils.u2 2) locals Attribute.Double_variable_info;
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DLOAD_3 ->
      check_load ofs (Utils.u2 3) locals Attribute.Double_variable_info;
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DMUL ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DNEG ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DREM ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DRETURN ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DSTORE parameter ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let locals = store (Utils.u2_of_u1 parameter) Attribute.Double_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.DSTORE_0 ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let locals = store (Utils.u2 0) Attribute.Double_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.DSTORE_1 ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let locals = store (Utils.u2 1) Attribute.Double_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.DSTORE_2 ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let locals = store (Utils.u2 2) Attribute.Double_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.DSTORE_3 ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let locals = store (Utils.u2 3) Attribute.Double_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.DSUB ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.DUP ->
      let v, stack = pop_if_category1 ofs stack in
      let stack = push v stack in
      let stack = push v stack in
      { locals = locals; stack = stack; }
  | Instruction.DUP2 ->
      let v1 = top ofs stack in
      let stack = pop ofs stack in
      let stack =
        if is_category1 v1 then
          let v2, stack = pop_if_category1 ofs stack in
          let stack = push v2 stack in
          let stack = push v1 stack in
          let stack = push v2 stack in
          push v1 stack
        else
          push v1 (push v1 stack) in
      { locals = locals; stack = stack; }
  | Instruction.DUP2_X1 ->
      let v1 = top ofs stack in
      let stack =
        if is_category1 v1 then
          let stack = pop ofs stack in
          let v2, stack = pop_if_category1 ofs stack in
          let v3, stack = pop_if_category1 ofs stack in
          let stack = push v2 stack in
          let stack = push v1 stack in
          let stack = push v3 stack in
          let stack = push v2 stack in
          push v1 stack
        else
          let stack = pop ofs stack in
          let v2, stack = pop_if_category1 ofs stack in
          let stack = push v1 stack in
          let stack = push v2 stack in
          push v1 stack in
      { locals = locals; stack = stack; }
  | Instruction.DUP2_X2 ->
      let v1 = top ofs stack in
      let stack =
        if is_category1 v1 then begin
          let stack = pop ofs stack in
          let v2, stack = pop_if_category1 ofs stack in
          let v3 = top ofs stack in
          if is_category1 v3 then begin
            let stack = pop ofs stack in
            let v4 = top ofs stack in
            let stack = pop ofs stack in
            let stack = push v2 stack in
            let stack = push v1 stack in
            let stack = push v4 stack in
            let stack = push v3 stack in
            let stack = push v2 stack in
            push v1 stack
          end else begin
            let stack = pop ofs stack in
            let stack = push v2 stack in
            let stack = push v1 stack in
            let stack = push v3 stack in
            let stack = push v2 stack in
            push v1 stack
          end
        end else begin
          let stack = pop ofs stack in
          let v2 = top ofs stack in
          if is_category1 v2 then begin
            let stack = pop ofs stack in
            let v3, stack = pop_if_category1 ofs stack in
            let stack = push v1 stack in
            let stack = push v3 stack in
            let stack = push v2 stack in
            push v1 stack
          end else begin
            let stack = pop ofs stack in
            let stack = push v1 stack in
            let stack = push v2 stack in
            push v1 stack
          end
        end in
      { locals = locals; stack = stack; }
  | Instruction.DUP_X1 ->
      let v1, stack = pop_if_category1 ofs stack in
      let v2, stack = pop_if_category1 ofs stack in
      let stack = push v1 stack in
      let stack = push v2 stack in
      let stack = push v1 stack in
      { locals = locals; stack = stack; }
  | Instruction.DUP_X2 ->
      let v1, stack = pop_if_category1 ofs stack in
      let v2 = top ofs stack in
      let stack =
        if is_category1 v2 then
          let v2, stack = pop_if_category1 ofs stack in
          let v3, stack = pop_if_category1 ofs stack in
          let stack = push v1 stack in
          let stack = push v3 stack in
          let stack = push v2 stack in
          push v1 stack
        else
          let v2, stack = pop_if_cat2 ofs stack in
          let stack = push v1 stack in
          let stack = push v2 stack in
          push v1 stack in
      { locals = locals; stack = stack; }
  | Instruction.F2D ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.F2I ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.F2L ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FADD ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FALOAD ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop ofs stack in
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FASTORE ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.FCMPG ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FCMPL ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FCONST_0 ->
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FCONST_1 ->
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FCONST_2 ->
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FDIV ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FLOAD parameter ->
      check_load ofs (Utils.u2_of_u1 parameter) locals Attribute.Float_variable_info;
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FLOAD_0 ->
      check_load ofs (Utils.u2 0) locals Attribute.Float_variable_info;
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FLOAD_1 ->
      check_load ofs (Utils.u2 1) locals Attribute.Float_variable_info;
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FLOAD_2 ->
      check_load ofs (Utils.u2 2) locals Attribute.Float_variable_info;
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FLOAD_3 ->
      check_load ofs (Utils.u2 3) locals Attribute.Float_variable_info;
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FMUL ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FNEG ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FREM ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FRETURN ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.FSTORE parameter ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let locals = store (Utils.u2_of_u1 parameter) Attribute.Float_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.FSTORE_0 ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let locals = store (Utils.u2 0) Attribute.Float_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.FSTORE_1 ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let locals = store (Utils.u2 1) Attribute.Float_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.FSTORE_2 ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let locals = store (Utils.u2 2) Attribute.Float_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.FSTORE_3 ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let locals = store (Utils.u2 3) Attribute.Float_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.FSUB ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.GETFIELD (_, _, desc) ->
      let topv = top ofs stack in
      check_reference ofs topv;
      let stack = pop ofs stack in
      let stack = push (Attribute.verification_type_info_of_parameter_descriptor desc) stack in
      { locals = locals; stack = stack; }
  | Instruction.GETSTATIC (_, _, desc) ->
      let stack = push (Attribute.verification_type_info_of_parameter_descriptor desc) stack in
      { locals = locals; stack = stack; }
  | Instruction.GOTO _ ->
      { locals = locals; stack = stack; }
  | Instruction.GOTO_W _ ->
      { locals = locals; stack = stack; }
  | Instruction.I2B ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.I2C ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.I2D ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.I2F ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.I2L ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.I2S ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IADD ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IALOAD ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop ofs stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IAND ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IASTORE ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.ICONST_0 ->
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ICONST_1 ->
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ICONST_2 ->
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ICONST_3 ->
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ICONST_4 ->
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ICONST_5 ->
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ICONST_M1 ->
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IDIV ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IF_ACMPEQ _ ->
      let stack = pop ofs stack in
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.IF_ACMPNE _ ->
      let stack = pop ofs stack in
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.IF_ICMPEQ _ ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IF_ICMPGE _ ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IF_ICMPGT _ ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IF_ICMPLE _ ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IF_ICMPLT _ ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IF_ICMPNE _ ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IFEQ _ ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IFGE _ ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IFGT _ ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IFLE _ ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IFLT _ ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IFNE _ ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IFNONNULL _ ->
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.IFNULL _ ->
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.IINC (parameter, _) ->
      check_load ofs (Utils.u2_of_u1 parameter) locals Attribute.Integer_variable_info;
      let locals = store (Utils.u2_of_u1 parameter) Attribute.Integer_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.ILOAD parameter ->
      check_load ofs (Utils.u2_of_u1 parameter) locals Attribute.Integer_variable_info;
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ILOAD_0 ->
      check_load ofs (Utils.u2 0) locals Attribute.Integer_variable_info;
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ILOAD_1 ->
      check_load ofs (Utils.u2 1) locals Attribute.Integer_variable_info;
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ILOAD_2 ->
      check_load ofs (Utils.u2 2) locals Attribute.Integer_variable_info;
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ILOAD_3 ->
      check_load ofs (Utils.u2 3) locals Attribute.Integer_variable_info;
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IMUL ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.INEG ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.INSTANCEOF _ ->
      let stack = pop ofs stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.INVOKEDYNAMIC (_, _, (params, ret)) ->
      let infos = List.rev_map Attribute.verification_type_info_of_parameter_descriptor params in
      let stack = List.fold_left (fun acc elem -> pop_if ofs elem acc) stack infos in
      let topv = top ofs stack in
      check_reference ofs topv;
      let stack = pop ofs stack in
      let stack = push_return_value ret stack in
      { locals = locals; stack = stack; }
  | Instruction.INVOKEINTERFACE (_, _, (params, ret)) ->
      let infos = List.rev_map Attribute.verification_type_info_of_parameter_descriptor params in
      let stack = List.fold_left (fun acc elem -> pop_if ofs elem acc) stack infos in
      let topv = top ofs stack in
      check_reference ofs topv;
      let stack = pop ofs stack in
      let stack = push_return_value ret stack in
      { locals = locals; stack = stack; }
  | Instruction.INVOKESPECIAL (cn, mn, (params, ret)) ->
      let infos = List.rev_map Attribute.verification_type_info_of_parameter_descriptor params in
      let stack = List.fold_left (fun acc elem -> pop_if ofs elem acc) stack infos in
      let topv = top ofs stack in
      check_reference ofs topv;
      let stack = pop ofs stack in
      let stack = push_return_value ret stack in
      let locals, stack =
        if UTF8.equal Consts.class_constructor (Name.utf8_for_method mn) then
          match topv with
          | Attribute.Uninitialized_variable_info ofs ->
              let f = function
                | Attribute.Uninitialized_variable_info ofs' when ofs = ofs' ->
                    Attribute.Object_variable_info (`Class_or_interface cn)
                | x -> x in
              let ff = function
                | Local x -> Local (f x)
                | x -> x in
              Array.map ff locals, List.map f stack
          | Attribute.Uninitialized_this_variable_info ->
              let f = function
                |  Attribute.Uninitialized_this_variable_info ->
                    Attribute.Object_variable_info (`Class_or_interface class_name)
                | x -> x in
              let ff = function
                | Local x -> Local (f x)
                | x -> x in
              Array.map ff locals, List.map f stack
          | _ -> locals, stack
        else
          locals, stack in
      { locals = locals; stack = stack; }
  | Instruction.INVOKESTATIC (_, _, (params, ret)) ->
      let infos = List.rev_map Attribute.verification_type_info_of_parameter_descriptor params in
      let stack = List.fold_left (fun acc elem -> pop_if ofs elem acc) stack infos in
      let stack = push_return_value ret stack in
      { locals = locals; stack = stack; }
  | Instruction.INVOKEVIRTUAL (_, _, (params, ret)) ->
      let infos = List.rev_map Attribute.verification_type_info_of_parameter_descriptor params in
      let stack = List.fold_left (fun acc elem -> pop_if ofs elem acc) stack infos in
      let topv = top ofs stack in
      check_reference ofs topv;
      let stack = pop ofs stack in
      let stack = push_return_value ret stack in
      { locals = locals; stack = stack; }
  | Instruction.IOR ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IREM ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IRETURN ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ISHL ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ISHR ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.ISTORE parameter ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let locals = store (Utils.u2_of_u1 parameter) Attribute.Integer_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.ISTORE_0 ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let locals = store (Utils.u2 0) Attribute.Integer_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.ISTORE_1 ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let locals = store (Utils.u2 1) Attribute.Integer_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.ISTORE_2 ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let locals = store (Utils.u2 2) Attribute.Integer_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.ISTORE_3 ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let locals = store (Utils.u2 3) Attribute.Integer_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.ISUB ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IUSHR ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.IXOR ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.JSR _ ->
      fail (Unsupported_instruction (ofs, "JSR"))
  | Instruction.JSR_W _ ->
      fail (Unsupported_instruction (ofs, "JSR_W"))
  | Instruction.L2D ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.L2F ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.L2I ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LADD ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LALOAD ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop ofs stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LAND ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LASTORE ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.LCMP ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LCONST_0 ->
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LCONST_1 ->
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LDC parameter ->
      let stack = push (verification_type_info_of_constant_descriptor parameter) stack in
      { locals = locals; stack = stack; }
  | Instruction.LDC2_W parameter ->
      let stack = push (verification_type_info_of_constant_descriptor parameter) stack in
      { locals = locals; stack = stack; }
  | Instruction.LDC_W parameter ->
      let stack = push (verification_type_info_of_constant_descriptor parameter) stack in
      { locals = locals; stack = stack; }
  | Instruction.LDIV ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LLOAD parameter ->
      check_load ofs (Utils.u2_of_u1 parameter) locals Attribute.Long_variable_info;
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LLOAD_0 ->
      check_load ofs (Utils.u2 0) locals Attribute.Long_variable_info;
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LLOAD_1 ->
      check_load ofs (Utils.u2 1) locals Attribute.Long_variable_info;
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LLOAD_2 ->
      check_load ofs (Utils.u2 2) locals Attribute.Long_variable_info;
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LLOAD_3 ->
      check_load ofs (Utils.u2 3) locals Attribute.Long_variable_info;
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LMUL ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LNEG ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LOOKUPSWITCH _ ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LOR ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LREM ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LRETURN ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LSHL ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LSHR ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LSTORE parameter ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let locals = store (Utils.u2_of_u1 parameter) Attribute.Long_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.LSTORE_0 ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let locals = store (Utils.u2 0) Attribute.Long_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.LSTORE_1 ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let locals = store (Utils.u2 1) Attribute.Long_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.LSTORE_2 ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let locals = store (Utils.u2 2) Attribute.Long_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.LSTORE_3 ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let locals = store (Utils.u2 3) Attribute.Long_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.LSUB ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LUSHR ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.LXOR ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.MONITORENTER ->
      let topv = top ofs stack in
      check_reference ofs topv;
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.MONITOREXIT ->
      let topv = top ofs stack in
      check_reference ofs topv;
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.MULTIANEWARRAY (at, dims) ->
      let s = ref stack in
      for _i = 1 to (dims :> int) do
        s := pop_if ofs Attribute.Integer_variable_info !s;
      done;
      let stack = push (Attribute.Object_variable_info at) !s in
      { locals = locals; stack = stack; }
  | Instruction.NEW _ ->
      let stack = push (Attribute.Uninitialized_variable_info ofs) stack in
      { locals = locals; stack = stack; }
  | Instruction.NEWARRAY parameter ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = push (verification_type_info_of_array_primitive ofs parameter) stack in
      { locals = locals; stack = stack; }
  | Instruction.NOP ->
      { locals = locals; stack = stack; }
  | Instruction.POP ->
      let _, stack = pop_if_category1 ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.POP2 ->
      let v1 = top ofs stack in
      let stack =
        if is_category1 v1 then
          snd (pop_if_category1 ofs (snd (pop_if_category1 ofs stack)))
        else
          pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.PUTFIELD (_, _, desc) ->
      let stack = pop_if ofs (Attribute.verification_type_info_of_parameter_descriptor desc) stack in
      let topv = top ofs stack in
      check_reference ofs topv;
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.PUTSTATIC (_, _, desc) ->
      let stack = pop_if ofs (Attribute.verification_type_info_of_parameter_descriptor desc) stack in
      { locals = locals; stack = stack; }
  | Instruction.RET _ ->
      fail (Unsupported_instruction (ofs, "RET"))
  | Instruction.RETURN ->
      { locals = locals; stack = stack; }
  | Instruction.SALOAD ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop ofs stack in
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.SASTORE ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let stack = pop ofs stack in
      { locals = locals; stack = stack; }
  | Instruction.SIPUSH _ ->
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.SWAP ->
      let v1, stack = pop_if_category1 ofs stack in
      let v2, stack = pop_if_category1 ofs stack in
      let stack = push v1 stack in
      let stack = push v2 stack in
      { locals = locals; stack = stack; }
  | Instruction.TABLESWITCH _ ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.WIDE_ALOAD parameter ->
      let loc = load ofs parameter locals in
      check_reference_local ofs loc;
      let stack = push_local loc stack in
      { locals = locals; stack = stack; }
  | Instruction.WIDE_ASTORE parameter ->
      let loc = top ofs stack in
      let stack = pop ofs stack in
      check_reference ofs loc;
      let locals = store parameter loc locals in
      { locals = locals; stack = stack; }
  | Instruction.WIDE_DLOAD parameter ->
      check_load ofs parameter locals Attribute.Double_variable_info;
      let stack = push Attribute.Double_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.WIDE_DSTORE parameter ->
      let stack = pop_if ofs Attribute.Double_variable_info stack in
      let locals = store parameter Attribute.Double_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.WIDE_FLOAD parameter ->
      check_load ofs parameter locals Attribute.Float_variable_info;
      let stack = push Attribute.Float_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.WIDE_FSTORE parameter ->
      let stack = pop_if ofs Attribute.Float_variable_info stack in
      let locals = store parameter Attribute.Float_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.WIDE_IINC (parameter, _) ->
      check_load ofs parameter locals Attribute.Integer_variable_info;
      let locals = store parameter Attribute.Integer_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.WIDE_ILOAD parameter ->
      check_load ofs parameter locals Attribute.Integer_variable_info;
      let stack = push Attribute.Integer_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.WIDE_ISTORE parameter ->
      let stack = pop_if ofs Attribute.Integer_variable_info stack in
      let locals = store parameter Attribute.Integer_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.WIDE_LLOAD parameter ->
      check_load ofs parameter locals Attribute.Long_variable_info;
      let stack = push Attribute.Long_variable_info stack in
      { locals = locals; stack = stack; }
  | Instruction.WIDE_LSTORE parameter ->
      let stack = pop_if ofs Attribute.Long_variable_info stack in
      let locals = store parameter Attribute.Long_variable_info locals in
      { locals = locals; stack = stack; }
  | Instruction.WIDE_RET _ ->
      fail (Unsupported_instruction (ofs, "WIDE_RET"))

type 'a unifier = 'a -> 'a -> 'a

type instance =
  [ `Array_type of Descriptor.array_type
  | `Class_or_interface of Name.for_class ]

let java_lang_Object_name = Name.make_for_class_from_external @"java.lang.Object"

let java_lang_Object = `Class_or_interface java_lang_Object_name

let make_array_unifier (f : Name.for_class unifier) (x : Descriptor.array_type) (y : Descriptor.array_type) =
  let rec ua (x : Descriptor.java_type) (y : Descriptor.java_type) =
    match x, y with
    | (`Array x'), (`Array y') -> `Array (ua (x' :> Descriptor.java_type) (y' :> Descriptor.java_type))
    | (`Array _), _ -> `Class java_lang_Object_name
    | _, (`Array _) -> `Class java_lang_Object_name
    | (`Class x'), (`Class y') -> `Class (f x' y')
    | `Boolean, `Boolean -> `Boolean
    | `Byte, `Byte -> `Byte
    | `Char, `Char -> `Char
    | `Double, `Double -> `Double
    | `Float, `Float -> `Float
    | `Int, `Int -> `Int
    | `Long, `Long -> `Long
    | `Short, `Short -> `Short
    | _ -> raise Not_found in
  try
    (match ua (x :> Descriptor.java_type) (y :> Descriptor.java_type) with
    | `Array x -> `Array_type (`Array x)
    | _ -> java_lang_Object)
  with Not_found -> java_lang_Object

let make_unifier f =
  let array_unifier = make_array_unifier f in
  fun x y ->
    if x == y then
      x
    else
      match x, y with
      | (`Array_type at1), (`Array_type at2) -> array_unifier at1 at2
      | (`Class_or_interface cn1), (`Class_or_interface cn2) -> `Class_or_interface (f cn1 cn2)
      | _ -> java_lang_Object

let unify_to_java_lang_Object =
  let utjlo x y =
    if (x == y) || (Name.equal_for_class x y) then
      x
    else
      java_lang_Object_name in
  make_unifier utjlo

module MemoTable = Hashtbl.Make (struct
  type t = Name.for_class
  let equal = Name.equal_for_class
  let hash = Name.hash_for_class
end)

let unify_to_closest_common_parent cl l =
  let rec parents cn =
    let hd, prn =
      try
        let c, p = List.find (fun (x, _) -> Name.equal_for_class cn x) l in
        c, p
      with Not_found ->
        let cd = ClassLoader.find_class_name cl cn in
        cd.ClassDefinition.name, cd.ClassDefinition.extends in
    let tl = match prn with
    | Some x -> parents x
    | None -> [] in
    hd :: tl in
  let memo_table = MemoTable.create 149 in
  let parents x =
    try
      MemoTable.find memo_table x
    with Not_found ->
      let res = parents x in
      MemoTable.add memo_table x res;
      res in
  let rec common_parent l = function
  | hd :: tl -> if List.exists (Name.equal_for_class hd) l then hd else common_parent l tl
  | [] -> java_lang_Object_name in
  let utccp x y =
    if x == y then
      x
    else
      let parents_x = parents x in
      let parents_y = parents y in
      common_parent parents_x parents_y in
  make_unifier utccp

let unify_to_parent_list l =
  let rec parents cn =
    let hd, prn =
      try
        let c, p = List.find (fun (x, _) -> Name.equal_for_class cn x) l in
        c, p
      with Not_found ->
        cn, None in
    let tl = match prn with
    | Some x -> parents x
    | None -> [] in
    hd :: tl in
  let rec common_parent l = function
  | hd :: tl -> if List.exists (Name.equal_for_class hd) l then hd else common_parent l tl
  | [] -> java_lang_Object_name in
  let utccp x y =
    if x == y then
      x
    else
      let parents_x = parents x in
      let parents_y = parents y in
      common_parent parents_x parents_y in
  make_unifier utccp

let unify f st1 st2 =
  let unify_elements vti1 vti2 =
    if vti1 == vti2 then
      vti1
    else
      match (vti1, vti2) with
      | Attribute.Top_variable_info, _
      | _, Attribute.Top_variable_info ->
          Attribute.Top_variable_info
      | (Attribute.Object_variable_info o1), (Attribute.Object_variable_info o2) ->
          Attribute.Object_variable_info (f o1 o2)
      | Attribute.Null_variable_info, (Attribute.Object_variable_info _) ->
          vti2
      | (Attribute.Object_variable_info _), Attribute.Null_variable_info ->
          vti1
      | _ ->
          if vti1 = vti2 then vti1 else Attribute.Top_variable_info in
  let shorten loc =
    let idx = ref (pred @@ Array.length loc) in
    while (!idx >= 0) && (loc.(!idx) = Local Attribute.Top_variable_info) do
      decr idx
    done;
    if !idx < 0 then
      [||]
    else
      Array.sub loc 0 (succ !idx) in
  let sz1 = List.length st1.stack in
  let sz2 = List.length st2.stack in
  if sz1 = sz2 then begin
    let len1 = Array.length st1.locals in
    let len2 = Array.length st2.locals in
    let len = Utils.max_int len1 len2 in
    let locals1 =
      Array.init
        len
        (fun i -> if i < len1 then st1.locals.(i) else Local Attribute.Top_variable_info) in
    let locals2 =
      Array.init
        len
        (fun i -> if i < len2 then st2.locals.(i) else Local Attribute.Top_variable_info) in
    let locals = Array.make len (Local Attribute.Top_variable_info) in
    let i = ref 0 in
    while !i < len do
      match locals1.(!i), locals2.(!i) with
      | Local Attribute.Long_variable_info, Local Attribute.Long_variable_info ->
          locals.(!i) <- Local Attribute.Long_variable_info;
          incr i;
          locals.(!i) <- Place_holder_long;
          incr i
      | Local Attribute.Double_variable_info, Local Attribute.Double_variable_info ->
          locals.(!i) <- Local Attribute.Double_variable_info;
          incr i;
          locals.(!i) <- Place_holder_double;
          incr i
      | Local Attribute.Long_variable_info, Local _
      | Local _, Local Attribute.Long_variable_info
      | Local Attribute.Double_variable_info, Local _
      | Local _, Local Attribute.Double_variable_info ->
          locals.(!i) <- Local Attribute.Top_variable_info;
          incr i;
          locals.(!i) <- Local Attribute.Top_variable_info;
          incr i
      | Local loc1, Local loc2 ->
          locals.(!i) <- Local (unify_elements loc1 loc2);
          incr i
      | Place_holder_long, _
      | Place_holder_double, _
      | _, Place_holder_long
      | _, Place_holder_double ->
          locals.(!i) <- Local Attribute.Top_variable_info;
          incr i
    done;
    let stack = List.map2 unify_elements st1.stack st2.stack in
    { locals = shorten locals; stack }
  end else
    fail (Different_stack_sizes (sz1, sz2))

let unify f st1 st2 =
  if (st1 == st2) then
    st1
  else
    unify f st1 st2

let encode ?(optimize = true) l =
  let to_list a =
    let res = ref [] in
    for i = pred (Array.length a) downto 0 do
      match a.(i) with
      | Local l ->
          res := l :: !res
      | Place_holder_long | Place_holder_double ->
          ()
    done;
    !res in
  let to_array a =
    a |> to_list |> Array.of_list in
  let full_frame ofs st =
    Attribute.Full_frame (ofs, (to_list st.locals), (List.rev st.stack)) in
  let encode2 ofs prev curr =
    if optimize && same_locals prev curr then
      match curr.stack with
      | [] -> Attribute.Same_frame ofs
      | [elem] -> Attribute.Same_locals_1_stack_item_frame (ofs, elem)
      | _ -> full_frame ofs curr
    else if optimize && curr.stack = [] then
      let curr_locals = to_array curr.locals in
      let prev_locals = to_array prev.locals in
      let curr_size = Array.length curr_locals in
      let prev_size = Array.length prev_locals in
      let eq = Attribute.equal_verification_type_info in
      match curr_size - prev_size with
      | -3 when array_for_all2 ~n:curr_size ~eq prev_locals curr_locals ->
          Attribute.Chop_3_frame ofs
      | -2 when array_for_all2 ~n:curr_size ~eq prev_locals curr_locals ->
          Attribute.Chop_2_frame ofs
      | -1 when array_for_all2 ~n:curr_size ~eq prev_locals curr_locals ->
          Attribute.Chop_1_frame ofs
      | 1 when array_for_all2 ~n:prev_size ~eq prev_locals curr_locals ->
          Attribute.Append_1_frame (ofs, curr_locals.(curr_size - 1))
      | 2 when array_for_all2 ~n:prev_size ~eq prev_locals curr_locals ->
          Attribute.Append_2_frame (ofs,
                                    curr_locals.(curr_size - 2),
                                    curr_locals.(curr_size - 1))
      | 3 when array_for_all2 ~n:prev_size ~eq prev_locals curr_locals ->
          Attribute.Append_3_frame (ofs,
                                    curr_locals.(curr_size - 3),
                                    curr_locals.(curr_size - 2),
                                    curr_locals.(curr_size - 1))
      | _ ->
          full_frame ofs curr
    else
      full_frame ofs curr in
  let _, res =
    match List.sort compare l with
    | hd :: tl ->
        List.fold_left
          (fun ((prev_ofs, prev_state), acc) (ofs, state) ->
            if prev_ofs <> ofs then
              (ofs, state), ((encode2 ofs prev_state state) :: acc)
            else if equal prev_state state then
              (prev_ofs, prev_state), acc
            else
              fail (Different_frames ofs))
          (hd, [full_frame (fst hd) (snd hd)])
          tl
    | [] ->
        fail Empty_frame_list in
  List.rev res
