(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

open Consts


(* Low-level form *)

type target =
  | Generic_type of type_parameter_target
  | Generic_method of type_parameter_target
  | Extends of supertype_target
  | Parameter_type of type_parameter_bound_target
  | Parameter_method of type_parameter_bound_target
  | Field
  | Return
  | Receiver
  | Formal_parameter of formal_parameter_target
  | Throws of throws_target
  | Local_variable of localvar_target
  | Resource_variable of localvar_target
  | Exception_ of catch_target
  | Instanceof of offset_target
  | New of offset_target
  | New_ref of offset_target
  | Method_ref of offset_target
  | Cast of type_argument_target
  | Generic_new_argument of type_argument_target
  | Generic_method_argument of type_argument_target
  | Generic_new_argument_ref of type_argument_target
  | Generic_method_argument_ref of type_argument_target
and type_parameter_target = Utils.u1
and supertype_target = Utils.u2
and type_parameter_bound_target = Utils.u1 * Utils.u1
and formal_parameter_target = Utils.u1
and throws_target = Utils.u2
and localvar_target = {
    start : Utils.u2;
    length : Utils.u2;
    index : Utils.u2;
  }
and catch_target = Utils.u2
and offset_target = Utils.u2
and type_argument_target = Utils.u2 * Utils.u1

type path_element =
  | Array_type
  | Nested_type
  | Wildcard_bound
  | Type_argument of Utils.u1

type path = path_element list

type primitive_type =
  [ `Boolean
  | `Byte
  | `Char
  | `Double
  | `Float
  | `Int
  | `Long
  | `Short ]

type info_element_value =
  | Primitive of primitive_type * Utils.u2
  | String of Utils.u2
  | Enum of Utils.u2 * Utils.u2
  | Class of Utils.u2
  | Annotation of info
  | Array of Utils.u2 * (info_element_value array)
and info = {
    type_index : Utils.u2;
    num_element_value_pairs : Utils.u2;
    element_value_pairs : (Utils.u2 * info_element_value) array;
  }
and extended_info = {
    ext_type_index : Utils.u2;
    ext_num_element_value_pairs : Utils.u2;
    ext_element_value_pairs : (Utils.u2 * info_element_value) array;
    ext_target : target;
    ext_path : path;
  }


(* Exception *)

BARISTA_ERROR =
  | Invalid_tag of (x : UChar.t) ->
      Printf.sprintf "invalid tag (%C)" (UChar.to_char_noerr x)
  | Inconsistent_primitive_value ->
      "inconsistent primitive value"
  | Invalid_string_value of (x : Utils.u2) ->
      Printf.sprintf "invalid string value (index %d)" (x :> int)
  | Invalid_enum_value of (x : Utils.u2) * (y : Utils.u2) ->
      Printf.sprintf "invalid enum value (indexes %d and %d)" (x :> int) (y :> int)
  | Invalid_class_value of (x : Utils.u2) ->
      Printf.sprintf "invalid class value (index %d)" (x :> int)
  | Invalid_annotation_type_value of (x : Utils.u2) ->
      Printf.sprintf "invalid annotation type value (index %d)" (x :> int)
  | Invalid_element_name of (x : Utils.u2) ->
      Printf.sprintf "invalid element name (index %d)" (x :> int)
  | Invalid_list_length ->
      Printf.sprintf "invalid list length"
  | Invalid_target of (x : Utils.u1) ->
      Printf.sprintf "invalid target (0x%02x)" (x :> int)
  | Invalid_path_kind of (x : Utils.u1) ->
      Printf.sprintf "invalid path kind (0x%02x)" (x :> int)
  | Invalid_path_argument of (x : Utils.u1) ->
      Printf.sprintf "invalid path argument (%d)" (x :> int)
  | Invalid_path_length ->
      Printf.sprintf "invalid path length"


(* I/O functions *)

let read_target st =
  let read_type_parameter_bound_target st =
    let x = InputStream.read_u1 st in
    let y = InputStream.read_u1 st in
    x, y in
  let read_localvar_target st =
    let start = InputStream.read_u2 st in
    let length = InputStream.read_u2 st in
    let index = InputStream.read_u2 st in
    { start; length; index } in
  let read_type_argument_target st =
    let x = InputStream.read_u2 st in
    let y = InputStream.read_u1 st in
    x, y in
  let target_type = InputStream.read_u1 st in
  match (target_type :> int) with
  | 0x00 -> Generic_type (InputStream.read_u1 st)
  | 0x01 -> Generic_method (InputStream.read_u1 st)
  | 0x10 -> Extends (InputStream.read_u2 st)
  | 0x11 -> Parameter_type (read_type_parameter_bound_target st)
  | 0x12 -> Parameter_method (read_type_parameter_bound_target st)
  | 0x13 -> Field
  | 0x14 -> Return
  | 0x15 -> Receiver
  | 0x16 -> Formal_parameter (InputStream.read_u1 st)
  | 0x17 -> Throws (InputStream.read_u2 st)
  | 0x40 -> Local_variable (read_localvar_target st)
  | 0x41 -> Resource_variable (read_localvar_target st)
  | 0x42 -> Exception_ (InputStream.read_u2 st)
  | 0x43 -> Instanceof (InputStream.read_u2 st)
  | 0x44 -> New (InputStream.read_u2 st)
  | 0x45 -> New_ref (InputStream.read_u2 st)
  | 0x46 -> Method_ref (InputStream.read_u2 st)
  | 0x47 -> Cast (read_type_argument_target st)
  | 0x48 -> Generic_new_argument (read_type_argument_target st)
  | 0x49 -> Generic_method_argument (read_type_argument_target st)
  | 0x4A -> Generic_new_argument_ref (read_type_argument_target st)
  | 0x4B ->  Generic_method_argument_ref (read_type_argument_target st)
  | _ -> fail (Invalid_target target_type)

let read_path_element st =
  let path_kind = InputStream.read_u1 st in
  let argument = InputStream.read_u1 st in
  if ((path_kind :> int) <= 2) && ((argument :> int) <> 0) then
    fail (Invalid_path_argument argument);
  match (path_kind :> int) with
  | 0 -> Array_type
  | 1 -> Nested_type
  | 2 -> Wildcard_bound
  | 3 -> Type_argument argument
  | _ -> fail (Invalid_path_kind path_kind)
    
let read_path st =
  let len = InputStream.read_u1 st in
  let res = ref [] in
  for _i = 1 to (len :> int) do
    res := (read_path_element st) :: !res
  done;
  List.rev !res

let write_target st t =
  let write_tag x =
    OutputStream.write_u1 st (Utils.u1 x) in
  let write_type_parameter_bound_target st (x, y) =
    OutputStream.write_u1 st x;
    OutputStream.write_u1 st y in
  let write_localvar_target st { start; length; index } =
    OutputStream.write_u2 st start;
    OutputStream.write_u2 st length;
    OutputStream.write_u2 st index in
  let write_type_argument_target st (x, y) =
    OutputStream.write_u2 st x;
    OutputStream.write_u1 st y in
  match t with
  | Generic_type x -> write_tag 0x00; OutputStream.write_u1 st x
  | Generic_method x -> write_tag 0x01; OutputStream.write_u1 st x
  | Extends x -> write_tag 0x10; OutputStream.write_u2 st x
  | Parameter_type x -> write_tag 0x11; write_type_parameter_bound_target st x
  | Parameter_method x -> write_tag 0x12; write_type_parameter_bound_target st x
  | Field -> write_tag 0x13
  | Return -> write_tag 0x14
  | Receiver -> write_tag 0x15
  | Formal_parameter x -> write_tag 0x16; OutputStream.write_u1 st x
  | Throws x -> write_tag 0x17; OutputStream.write_u2 st x
  | Local_variable x -> write_tag 0x40; write_localvar_target st x
  | Resource_variable x -> write_tag 0x41; write_localvar_target st x
  | Exception_ x -> write_tag 0x42; OutputStream.write_u2 st x
  | Instanceof x -> write_tag 0x43; OutputStream.write_u2 st x
  | New x -> write_tag 0x44; OutputStream.write_u2 st x
  | New_ref x -> write_tag 0x45; OutputStream.write_u2 st x
  | Method_ref x -> write_tag 0x46; OutputStream.write_u2 st x
  | Cast x -> write_tag 0x47; write_type_argument_target st x
  | Generic_new_argument x -> write_tag 0x48; write_type_argument_target st x
  | Generic_method_argument x -> write_tag 0x49; write_type_argument_target st x
  | Generic_new_argument_ref x -> write_tag 0x4A; write_type_argument_target st x
  | Generic_method_argument_ref x -> write_tag 0x4B; write_type_argument_target st x

let write_path_element st pe =
  let write st x y =
    OutputStream.write_u1 st (Utils.u1 x);
    OutputStream.write_u1 st y in
  match pe with
  | Array_type -> write st 0 (Utils.u1 0)
  | Nested_type -> write st 1 (Utils.u1 0)
  | Wildcard_bound -> write st 2 (Utils.u1 0)
  | Type_argument x -> write st 3 x

let write_path st p =
  let len = List.length p in
  if len > 255 then fail Invalid_path_length;
  OutputStream.write_u1 st (Utils.u1 len);
  List.iter (fun pe -> write_path_element st pe) p

let rec read_info_element_value st =
  let tag = UChar.of_code ((InputStream.read_u1 st) :> int) in
  let primitive p _ =
    let index = InputStream.read_u2 st in
    Primitive (p, index) in
  Utils.switch UChar.equal
    [ capital_b, primitive `Byte;
      capital_c, primitive `Char;
      capital_d, primitive `Double;
      capital_f, primitive `Float;
      capital_i, primitive `Int;
      capital_j, primitive `Long;
      capital_s, primitive `Short;
      capital_z, primitive `Boolean;
      small_s,
      (fun _ ->
        let index = InputStream.read_u2 st in
        String index);
      small_e,
      (fun _ ->
        let type_name_index = InputStream.read_u2 st in
        let const_name_index = InputStream.read_u2 st in
        Enum (type_name_index, const_name_index));
      small_c,
      (fun _ ->
        let class_info_index = InputStream.read_u2 st in
        Class class_info_index);
      at_character,
      (fun _ ->
        let annot = read_info st in
        Annotation annot);
      opening_square_bracket,
      (fun _ ->
        let num_values = InputStream.read_u2 st in
        let values = Array.init (num_values :> int) (fun _ -> read_info_element_value st) in
        Array (num_values, values)) ]
    (fun tag -> fail (Invalid_tag tag))
    tag
and read_info st =
  let type_idx = InputStream.read_u2 st in
  let nb = InputStream.read_u2 st in
  let evp =
    Array.init
      (nb :> int)
      (fun _ ->
        let name_index = InputStream.read_u2 st in
        let el_vl = read_info_element_value st in
        (name_index, el_vl)) in
  { type_index = type_idx;
    num_element_value_pairs = nb;
    element_value_pairs = evp; }
and read_extended_info st =
  let t = read_target st in
  let p = read_path st in
  let type_idx = InputStream.read_u2 st in
  let nb = InputStream.read_u2 st in
  let evp =
    Array.init
      (nb :> int)
      (fun _ ->
        let name_index = InputStream.read_u2 st in
        let el_vl = read_info_element_value st in
        (name_index, el_vl)) in
  { ext_type_index = type_idx;
    ext_num_element_value_pairs = nb;
    ext_element_value_pairs = evp;
    ext_target = t;
    ext_path = p; }

let rec write_info_element_value st i =
  match i with
  | Primitive (jt, idx) ->
      let ch = match jt with
      | `Boolean -> capital_z
      | `Byte -> capital_b
      | `Char -> capital_c
      | `Double -> capital_d
      | `Float -> capital_f
      | `Int -> capital_i
      | `Long -> capital_l
      | `Short -> capital_s in
      OutputStream.write_u1 st (Utils.u1 (UChar.to_code ch));
      OutputStream.write_u2 st idx
  | String idx ->
      OutputStream.write_u1 st (Utils.u1 (UChar.to_code small_s));
      OutputStream.write_u2 st idx
  | Enum (n, v) ->
      OutputStream.write_u1 st (Utils.u1 (UChar.to_code small_e));
      OutputStream.write_u2 st n;
      OutputStream.write_u2 st v
  | Class idx ->
      OutputStream.write_u1 st (Utils.u1 (UChar.to_code small_c));
      OutputStream.write_u2 st idx
  | Annotation i ->
      OutputStream.write_u1 st (Utils.u1 (UChar.to_code at_character));
      write_info st i
  | Array (len, arr) ->
      OutputStream.write_u1 st (Utils.u1 (UChar.to_code opening_square_bracket));
      OutputStream.write_u2 st len;
      Array.iter (fun x -> write_info_element_value st x) arr
and write_info st i =
  OutputStream.write_u2 st i.type_index;
  OutputStream.write_u2 st i.num_element_value_pairs;
  Array.iter
    (fun (idx, iev) ->
      OutputStream.write_u2 st idx;
      write_info_element_value st iev)
    i.element_value_pairs
and write_extended_info st i =
  write_target st i.ext_target;
  write_path st i.ext_path;
  OutputStream.write_u2 st i.ext_type_index;
  OutputStream.write_u2 st i.ext_num_element_value_pairs;
  Array.iter
    (fun (idx, iev) ->
      OutputStream.write_u2 st idx;
      write_info_element_value st iev)
    i.ext_element_value_pairs


(* High-level form *)

type element_value =
  | Boolean_value of bool
  | Byte_value of int
  | Char_value of UChar.t
  | Double_value of float
  | Float_value of float
  | Int_value of int32
  | Long_value of int64
  | Short_value of int
  | String_value of UTF8.t
  | Enum_value of Name.for_class * Name.for_field
  | Class_value of Name.for_class
  | Annotation_value of t
  | Array_value of element_value list
and t = Name.for_class * ((UTF8.t * element_value) list)
and extended = Name.for_class * ((UTF8.t * element_value) list) * target * path

let rec equal_element_value x y =
  match x, y with
  | (Boolean_value b1), (Boolean_value b2) -> b1 = b2
  | (Byte_value b1), (Byte_value b2) -> b1 = b2
  | (Char_value c1), (Char_value c2) -> c1 = c2
  | (Double_value d1), (Double_value d2) -> d1 = d2
  | (Float_value f1), (Float_value f2) -> f1 = f2
  | (Int_value i1), (Int_value i2) -> i1 = i2
  | (Long_value l1), (Long_value l2) -> l1 = l2
  | (Short_value s1), (Short_value s2) -> s1 = s2
  | (String_value s1), (String_value s2) -> UTF8.equal s1 s2
  | (Enum_value (cn1, fn1)), (Enum_value (cn2, fn2)) -> (Name.equal_for_class cn1 cn2) && (Name.equal_for_field fn1 fn2)
  | (Class_value cn1), (Class_value cn2) -> Name.equal_for_class cn1 cn2
  | (Annotation_value a1), (Annotation_value a2) -> equal a1 a2
  | (Array_value a1), (Array_value a2) -> Utils.list_equal equal_element_value a1 a2
  | _ -> false
and equal (cn1, l1) (cn2, l2) =
  (Name.equal_for_class cn1 cn2)
    && (Utils.list_equal equal_utf8_element l1 l2)
and equal_extended (cn1, l1, t1, p1) (cn2, l2, t2, p2) =
  (Name.equal_for_class cn1 cn2)
    && (Utils.list_equal equal_utf8_element l1 l2)
    && (t1 = t2)
    && (Utils.list_equal Pervasives.(=) p1 p2)
and equal_utf8_element (u1, e1) (u2, e2) =
    (UTF8.equal u1 u2) && (equal_element_value e1 e2)

let rec compare_element_value x y =
  match x, y with
  | (Boolean_value b1), (Boolean_value b2) -> Pervasives.compare b1 b2
  | (Byte_value b1), (Byte_value b2) -> Pervasives.compare b1 b2
  | (Char_value c1), (Char_value c2) -> Pervasives.compare c1 c2
  | (Double_value d1), (Double_value d2) -> Pervasives.compare d1 d2
  | (Float_value f1), (Float_value f2) -> Pervasives.compare f1 f2
  | (Int_value i1), (Int_value i2) -> Pervasives.compare i1 i2
  | (Long_value l1), (Long_value l2) -> Pervasives.compare l1 l2
  | (Short_value s1), (Short_value s2) -> Pervasives.compare s1 s2
  | (String_value s1), (String_value s2) -> UTF8.compare s1 s2
  | (Enum_value (cn1, fn1)), (Enum_value (cn2, fn2)) ->
      let res = Name.compare_for_class cn1 cn2 in
      if res = 0 then
        Name.compare_for_field fn1 fn2
      else
        res
  | (Class_value cn1), (Class_value cn2) -> Name.compare_for_class cn1 cn2
  | (Annotation_value a1), (Annotation_value a2) -> compare a1 a2
  | (Array_value a1), (Array_value a2) -> Utils.list_compare compare_element_value a1 a2
  | _ -> Pervasives.compare x y
and compare (cn1, l1) (cn2, l2) =
  let res = Name.compare_for_class cn1 cn2 in
  if res = 0 then
    Utils.list_compare compare_utf8_element l1 l2
  else
    res
and compare_extended (cn1, l1, t1, p1) (cn2, l2, t2, p2) =
  let res = Name.compare_for_class cn1 cn2 in
  if res = 0 then
    let res' = Utils.list_compare compare_utf8_element l1 l2 in
    if res' = 0 then
      let res'' = Pervasives.compare t1 t2 in
      if res'' = 0 then
        Utils.list_compare Pervasives.compare p1 p2
      else
        res''
    else
      res'
  else
    res
and compare_utf8_element (u1, e1) (u2, e2) =
  let res = UTF8.compare u1 u2 in
  if res = 0 then
    compare_element_value e1 e2
  else
    res

let rec hash_element_value = function
  | Boolean_value _ -> 1
  | Byte_value _ -> 2
  | Char_value _ -> 3
  | Double_value _ -> 4
  | Float_value _ -> 5
  | Int_value _ -> 6
  | Long_value _ -> 7
  | Short_value _ -> 8
  | String_value _ -> 9
  | Enum_value _ -> 10
  | Class_value _ -> 11
  | Annotation_value _ -> 12
  | Array_value _ -> 13
and hash (cn, l) =
  (Name.hash_for_class cn) + (Utils.list_hash hash_utf8_element l)
and hash_extended (cn, l, _, _) =
  (Name.hash_for_class cn) + (Utils.list_hash hash_utf8_element l)
and hash_utf8_element (x, y) =
  (UTF8.hash x) + (hash_element_value y)


(* Conversion functions *)

let rec decode_element_value pool i =
  match i with
  | Primitive (d, idx) ->
      (match (ConstantPool.get_entry pool idx), d with
      | (ConstantPool.Integer v), `Boolean ->
          Boolean_value (v <> 0l)
      | (ConstantPool.Integer v), `Byte ->
          Byte_value (Int32.to_int v)
      | (ConstantPool.Integer v), `Char ->
          Char_value (UChar.of_code (Int32.to_int v))
      | (ConstantPool.Double (hi, lo)), `Double ->
          let x = Utils.(gather_s8 (s4 hi) (s4 lo)) in
          Double_value (Int64.float_of_bits (x :> int64))
      | (ConstantPool.Float f), `Float ->
          Float_value (Int32.float_of_bits f)
      | (ConstantPool.Integer v), `Int ->
          Int_value v
      | (ConstantPool.Long (hi, lo)), `Long ->
          let x = Utils.(gather_s8 (s4 hi) (s4 lo)) in
          Long_value (x :> int64)
      | (ConstantPool.Integer v), `Short ->
          Short_value (Int32.to_int v)
      | _ -> fail Inconsistent_primitive_value)
  | String idx ->
      (match ConstantPool.get_entry pool idx with
      | ConstantPool.UTF8 n -> String_value n
      | _ -> fail (Invalid_string_value idx))
  | Enum (name_idx, value_idx) ->
      (match (ConstantPool.get_entry pool name_idx),
        (ConstantPool.get_entry pool value_idx) with
      | (ConstantPool.UTF8 n), (ConstantPool.UTF8 v) ->
          (match Descriptor.field_of_utf8 n with
          | `Class cn ->
              Enum_value (cn, (Name.make_for_field v))
          | _ -> fail (Invalid_enum_value (name_idx, value_idx)))
      | _ -> fail (Invalid_enum_value (name_idx, value_idx)))
  | Class idx ->
      (match ConstantPool.get_entry pool idx with
      | ConstantPool.UTF8 n ->
          (match Descriptor.java_type_of_internal_utf8 n with
          | `Class x -> Class_value x
          | _ -> fail (Invalid_class_value idx))
      | _ -> fail (Invalid_class_value idx))
  | Annotation i ->
      Annotation_value (decode pool i)
  | Array (_, arr) ->
      Array_value (List.map (decode_element_value pool) (Array.to_list arr))
and decode pool i =
    let type_desc = match ConstantPool.get_entry pool i.type_index with
    | ConstantPool.UTF8 n ->
        (match Descriptor.java_type_of_internal_utf8 n with
        | `Class c -> c
        | _ -> fail (Invalid_annotation_type_value i.type_index))
    | _ -> fail (Invalid_annotation_type_value i.type_index) in
    let pairs = List.map (fun (idx, iev) ->
      let name = match ConstantPool.get_entry pool idx with
      | ConstantPool.UTF8 n -> n
      | _ -> fail (Invalid_element_name idx) in
      (name, (decode_element_value pool iev)))
        (Array.to_list i.element_value_pairs) in
    (type_desc, pairs)

let decode_extended pool i =
  let x, y =
    decode
      pool
      { type_index = i.ext_type_index;
        num_element_value_pairs = i.ext_num_element_value_pairs;
        element_value_pairs = i.ext_element_value_pairs; } in
  x, y, i.ext_target, i.ext_path

let rec encode_element_value pool e =
  match e with
  | Boolean_value v ->
      let idx = ConstantPool.add_integer pool (if v then 1l else 0l) in
      Primitive (`Boolean, idx)
  | Byte_value v ->
      let idx = ConstantPool.add_integer pool (Int32.of_int v) in
      Primitive (`Byte, idx)
  | Char_value v ->
      let idx = ConstantPool.add_integer pool (Int32.of_int (UChar.to_code v)) in
      Primitive (`Char, idx)
  | Double_value v ->
      let idx = ConstantPool.add_double pool v in
      Primitive (`Double, idx)
  | Float_value v ->
      let idx = ConstantPool.add_float pool v in
      Primitive (`Float, idx)
  | Int_value v ->
      let idx = ConstantPool.add_integer pool v in
      Primitive (`Int, idx)
  | Long_value v ->
      let idx = ConstantPool.add_long pool v in
      Primitive (`Long, idx)
  | Short_value v ->
      let idx = ConstantPool.add_integer pool (Int32.of_int v) in
      Primitive (`Short, idx)
  | String_value v ->
      let idx = ConstantPool.add_utf8 pool v in
      String idx
  | Enum_value (n, v) ->
      let n' = (Descriptor.utf8_of_field (`Class n)) in
      let n_idx = ConstantPool.add_utf8 pool n' in
      let v_idx = ConstantPool.add_utf8 pool (Name.utf8_for_field v) in
      Enum (n_idx, v_idx)
  | Class_value n ->
      let desc = Descriptor.internal_utf8_of_java_type (`Class n) in
      let idx = ConstantPool.add_utf8 pool desc in
      Class idx
  | Annotation_value a ->
      Annotation (encode pool a)
  | Array_value l ->
      let arr = Array.of_list (List.map (encode_element_value pool) l) in
      Array (Utils.u2 (Array.length arr), arr)
and encode pool e =
  let (field, pairs) = e in
  let field_value = Descriptor.utf8_of_field (`Class field) in
  let field_index = ConstantPool.add_utf8 pool field_value in
  let pairs' = List.map (fun (n, v) ->
    let n_idx = ConstantPool.add_utf8 pool n in
    (n_idx, (encode_element_value pool v))) pairs in
  let len_pairs = List.length pairs in
  if len_pairs > Utils.max_u2_value then
    fail Invalid_list_length;
  { type_index = field_index;
    num_element_value_pairs = Utils.u2 len_pairs;
    element_value_pairs = Array.of_list pairs'; }

let encode_extended pool e =
  let x, y, z, t = e in
  let i = encode pool (x, y) in
  { ext_type_index = i.type_index;
    ext_num_element_value_pairs = i.num_element_value_pairs;
    ext_element_value_pairs = i.element_value_pairs;
    ext_target = z;
    ext_path = t; }
