(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

let (++) = UTF8.(++)


(* Types *)

type for_field = Name.for_class * Name.for_field * Descriptor.for_field

type for_method = Name.for_class * Name.for_method * Descriptor.for_method

type for_constructor = Name.for_class * (Descriptor.for_parameter list)

type t =
  | GetField of for_field
  | GetStatic of for_field
  | PutField of for_field
  | PutStatic of for_field
  | InvokeVirtual of for_method
  | InvokeStatic of for_method
  | InvokeSpecial of for_method
  | NewInvokeSpecial of for_constructor
  | InvokeInterface of for_method

type kind =
  | REF_getField
  | REF_getStatic
  | REF_putField
  | REF_putStatic
  | REF_invokeVirtual
  | REF_invokeStatic
  | REF_invokeSpecial
  | REF_newInvokeSpecial
  | REF_invokeInterface


(* Exception *)

BARISTA_ERROR =
  | Invalid_kind of (x : Utils.u1) ->
      Printf.sprintf "invalid reference kind (%d)" (x :> int)


(* Functions *)

let u1_of_kind x =
  let res = match x with
  | REF_getField -> 1
  | REF_getStatic -> 2
  | REF_putField -> 3
  | REF_putStatic -> 4
  | REF_invokeVirtual -> 5
  | REF_invokeStatic -> 6
  | REF_invokeSpecial -> 7
  | REF_newInvokeSpecial -> 8
  | REF_invokeInterface -> 9 in
  Utils.u1 res

let kind_of_u1 x = 
  match (x : Utils.u1 :> int) with
  | 1 -> REF_getField
  | 2 -> REF_getStatic
  | 3 -> REF_putField
  | 4 -> REF_putStatic
  | 5 -> REF_invokeVirtual
  | 6 -> REF_invokeStatic
  | 7 -> REF_invokeSpecial
  | 8 -> REF_newInvokeSpecial
  | 9 -> REF_invokeInterface
  | x -> fail (Invalid_kind (Utils.u1 x))

let utf8_for_field (cn, fn, fd) =
  (Name.external_utf8_for_class cn)
    ++ @"."
    ++ (Name.utf8_for_field fn)
    ++ @":"
    ++ (Descriptor.utf8_of_field fd)

let utf8_for_method (cn, mn, md) =
  (Name.external_utf8_for_class cn)
    ++ @"."
    ++ (Name.utf8_for_method mn)
    ++ (Descriptor.utf8_of_method md)

let utf8_for_constructor (cn, pl) =
  (Name.external_utf8_for_class cn)
    ++ @"("
    ++ (UTF8.concat (List.map Descriptor.utf8_of_parameter pl))
    ++ @")"

let to_utf8 = function
  | GetField f -> @"GetField " ++ (utf8_for_field f)
  | GetStatic f -> @"GetStatic " ++ (utf8_for_field f)
  | PutField f -> @"PutField " ++ (utf8_for_field f)
  | PutStatic f -> @"PutStatic " ++ (utf8_for_field f)
  | InvokeVirtual m -> @"InvokeVirtual " ++ (utf8_for_method m)
  | InvokeStatic m -> @"" ++ (utf8_for_method m)
  | InvokeSpecial m -> @"InvokeStatic " ++ (utf8_for_method m)
  | NewInvokeSpecial c -> @"NewInvokeSpecial " ++ (utf8_for_constructor c)
  | InvokeInterface m -> @"InvokeInterface " ++ (utf8_for_method m)

let equal_for_field (cn1, fn1, fd1) (cn2, fn2, fd2) =
  (Name.equal_for_class cn1 cn2)
    && (Name.equal_for_field fn1 fn2)
    && (Descriptor.equal_for_field fd1 fd2)

let equal_for_method (cn1, mn1, md1) (cn2, mn2, md2) =
  (Name.equal_for_class cn1 cn2)
    && (Name.equal_for_method mn1 mn2)
    && (Descriptor.equal_for_method md1 md2)

let equal_for_constructor (cn1, pl1) (cn2, pl2) =
  (Name.equal_for_class cn1 cn2)
    && (Utils.list_equal Descriptor.equal_for_parameter pl1 pl2)

let equal x y =
  match x, y with
  | GetField f1, GetField f2
  | GetStatic f1, GetStatic f2
  | PutField f1, PutField f2
  | PutStatic f1, PutStatic f2 ->
      equal_for_field f1 f2
  | InvokeVirtual m1, InvokeVirtual m2
  | InvokeStatic m1, InvokeStatic m2
  | InvokeSpecial m1, InvokeSpecial m2 
  | InvokeInterface m1, InvokeInterface m2 ->
      equal_for_method m1 m2
  | NewInvokeSpecial c1, NewInvokeSpecial c2 ->
      equal_for_constructor c1 c2
  | _ -> false

let compare_for_field (cn1, fn1, fd1) (cn2, fn2, fd2) =
  let res = Name.compare_for_class cn1 cn2 in
  if res = 0 then
    let res' = Name.compare_for_field fn1 fn2 in
    if res' = 0 then
      Descriptor.compare_for_field fd1 fd2
    else
      res'
  else
    res

let compare_for_method (cn1, mn1, md1) (cn2, mn2, md2) =
  let res = Name.compare_for_class cn1 cn2 in
  if res = 0 then
    let res' = Name.compare_for_method mn1 mn2 in
    if res' = 0 then
      Descriptor.compare_for_method md1 md2
    else
      res'
  else
    res

let compare_for_constructor (cn1, pl1) (cn2, pl2) =
  let res = Name.compare_for_class cn1 cn2 in
  if res = 0 then
    Utils.list_compare Descriptor.compare_for_parameter pl1 pl2
  else
    res

let compare x y =
  match x, y with
  | GetField f1, GetField f2
  | GetStatic f1, GetStatic f2
  | PutField f1, PutField f2
  | PutStatic f1, PutStatic f2 ->
      compare_for_field f1 f2
  | InvokeVirtual m1, InvokeVirtual m2
  | InvokeStatic m1, InvokeStatic m2
  | InvokeSpecial m1, InvokeSpecial m2 
  | InvokeInterface m1, InvokeInterface m2 ->
      compare_for_method m1 m2
  | NewInvokeSpecial c1, NewInvokeSpecial c2 ->
      compare_for_constructor c1 c2
  | _ -> Pervasives.compare x y

let hash_for_field (cn, fn, fd) =
  (Name.hash_for_class cn)
    + (Name.hash_for_field fn)
    + (Descriptor.hash_for_field fd)

let hash_for_method (cn, mn, md) =
  (Name.hash_for_class cn)
    + (Name.hash_for_method mn)
    + (Descriptor.hash_for_method md)

let hash_for_constructor (cn, pl) =
  (Name.hash_for_class cn)
    + (Utils.list_hash Descriptor.hash_for_parameter pl)

let hash = function
  | GetField f -> 1 + (hash_for_field f)
  | GetStatic f -> 2 + (hash_for_field f)
  | PutField f -> 3 + (hash_for_field f)
  | PutStatic f -> 4 + (hash_for_field f)
  | InvokeVirtual m -> 5 + (hash_for_method m)
  | InvokeStatic m -> 6 + (hash_for_method m)
  | InvokeSpecial m -> 7 + (hash_for_method m)
  | NewInvokeSpecial c -> 8 + (hash_for_constructor c)
  | InvokeInterface m -> 9 + (hash_for_method m)
