(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Abstraction over class path elements. *)


(** {6 Types} *)

type t
(** The type of class paths.*)

BARISTA_ERROR =
  | Unable_to_open_archive of Path.t
  | Does_not_exist of Path.t
  | Class_not_found of UTF8.t


(** {6 Constructors} *)

val default_separator : UChar.t
(** Default separator between classpath elements.
    Loaded from the [BARISTA_CLASSPATH_SEPARATOR] environment variable if
    set, heuristically determined otherwise. *)

val make_of_utf8 : ?separator:UChar.t -> UTF8.t -> t
(** Constructs a class path from a string, using the passed separator
    between class path elements (defaulting to [default_separator]).

    Raises [Exception] if a class path element does not exists or if an
    archive cannot be opened. *)

val make_of_path_list : Path.t list -> t
(** Constructs a class path from the passed elements.

    Raises [Exception] if a class path element does not exists or if an
    archive cannot be opened. *)

val make_of_utf8_list : UTF8.t list -> t
(** Constructs a class path from the passed elements.

    Raises [Exception] if a class path element does not exists or if an
    archive cannot be opened. *)

val get_default : unit -> Path.t list
(** Returns the list of elements to be used by [make]. *)

val make : unit -> t
(** Constructs a class path from the ["CLASSPATH"] environment variable,
    defaulting to ["."] if the variable is not set.

    {b Does not raise an exception if a class path element does not
    exists or if an archive cannot be opened.} *)

val make_empty : unit -> t
(** Constructs a class path with no element. *)

val append : ?separator:UChar.t -> UTF8.t -> t -> t
(** [append str cp] returns a class path that is [cp] plus [str] added at its
    end. The [separator] parameter defaults to [default_separator].

    Raises [Exception] if a class path element does not exists or if an
    archive cannot be opened. *)

val append_path : Path.t -> t -> t
(** [append path cp] returns a class path that is [cp] plus [path] added at its
    end.

    Raises [Exception] if passed element cannot be opened. *)

val prepend : ?separator:UChar.t -> UTF8.t -> t -> t
(** [prepend str cp] returns a class path that is [cp] plus [str] added at its
    begin. The [separator] parameter defaults to [default_separator].

    Raises [Exception] if a class path element does not exists or if an
    archive cannot be opened. *)

val prepend_path : Path.t -> t -> t
(** [prepend path cp] returns a class path that is [cp] plus [path] added at its
    begin.

    Raises [Exception] if passed element cannot be opened. *)


(** {6 Functions} *)

val open_stream : t -> UTF8.t -> InputStream.t
(** [open_stream cp cn] returns the stream for class whose fully qualified
    name is [cn], search class path [cp].

    Raises [Exception] if the passed class is not found in the given class
    path. *)

val close : t -> unit
(** Closes all underlying archives.

    Raises an exception if an archive cannot be closed. *)

val close_noerr : t -> unit
(** Same as [close] but raised exceptions are silently discarded. *)

val equal : t -> t -> bool
(** Equality over class paths. *)

val compare : t -> t -> int
(** Comparison over class paths. *)

val hash : t -> int
(** Hash function over class paths. *)

val to_utf8 : t -> UTF8.t
(** Converts the passed class path into a string. *)

val to_internal_utf8 : t -> UTF8.t
(** Converts the passed class path into a string (using internal/debug
    format). *)
