(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Support for creation of manifest files. *)

type version =
  | Version_1_0 (** Version 1.0. *)
(** The type representing the various versions of the manifest file
    format. *)

type custom_creator = {
    creator_name : UTF8.t; (** Name of the software creating the manifest. *)
    creator_version : UTF8.t; (** Version of the software creating the manifest. *)
  }
(** The type representing custom creators. *)

type creator =
  | Barista_creator
      (** Used to indicate that Barista created the manifest. *)
  | Custom_creator of custom_creator
      (** Used to indicate a custom creator. *)
  | No_creator
      (** Used to elide creator information. *)
(** The type representing manifest creator. *)

type required_extension = {
    rext_short_name : UTF8.t;
        (** Extension short name (e.g. ["java3d"]. *)
    rext_full_name : UTF8.t;
        (** Extension full name (e.g. ["javax.3d"]. *)
    rext_spec_vendor : UTF8.t;
        (** Specification vendor (e.g. ["Company name"]. *)
    rext_spec_version : UTF8.t;
        (** Specification version (e.g. ["1.0"]. *)
    rext_impl_vendor_id : UTF8.t;
        (** Implementation vendor identifier (e.g. ["com.company"]. *)
    rext_impl_vendor : UTF8.t;
        (** Implementation vendor (e.g. ["Company name"]. *)
    rext_impl_version : UTF8.t;
        (** Implementation version (e.g. ["1.0"]. *)
    rext_impl_url : UTF8.t;
        (** Implementation URL (e.g. ["http://company.com/impl"]. *)
  }
(** The type of required extensions. *)

type provided_extension = {
    pext_name : UTF8.t;
        (** Extension name, that is identifier. *)
    pext_impl_title : UTF8.t;
        (** Implementation title (e.g. ["Java 3D"]). *)
    pext_impl_version : UTF8.t;
        (** Implementation version (e.g. ["1.0"]. *)
    pext_impl_vendor : UTF8.t;
        (** Implementation vendor (e.g. ["Company name"]. *)
    pext_impl_vendor_id : UTF8.t;
        (** Implementation vendor identifier (e.g. ["com.company"]. *)
    pext_spec_title : UTF8.t;
        (** Specification title (e.g. ["Java 3D"]). *)
    pext_spec_version : UTF8.t;
        (** Specification version (e.g. ["1.0"]. *)
    pext_spec_vendor : UTF8.t;
        (** Specification vendor (e.g. ["Company name"]. *)
  }
(** The type of provided extensions. *)

type entry = {
    entry_path : UTF8.t;
        (** Path to entry. *)
    entry_type : UTF8.t option;
        (** MIME type of entry. *)
    entry_bean : bool option;
        (** Whether the entry designates a Java Bean. *)
    entry_sealed : bool;
        (** Whether the entry is sealed. *)
  }
(** The type of entry-specific information. *)

type t = {
    version : version;
        (** Manifest version. *)
    creator : creator;
        (** Manifest creator. *)
    sealed : bool;
        (** Whether the archive is sealed. *)
    main_class : Name.for_class option;
        (** Main class name, if any (for executable archive). *)
    class_path : UTF8.t list;
        (** Class path. *)
    extensions : required_extension list;
        (** List of required extensions. *)
    extension : provided_extension option;
        (** Extension implemented by archive, if any. *)
    entries : entry list;
        (** List of per-path entries. *)
  }
(** The type representing the contents of a manifest file. *)

val path_in_archive : UTF8.t
(** The path of a manifest file inside an archive. *)

val default : t
(** Default manifest, with minimal information:
    - Barista as the creator;
    - not sealed;
    - no main class;
    - empty class path;
    - no required extension;
    - no provided extension;
    - no per-path entry. *)

val to_lines : t -> UTF8.t list
(** Converts the passed manifest into a list of lines. *)

val to_utf8 : t -> UTF8.t
(** Converts the passed manifest into a bare string. *)
