(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

IFNDEF USE_JDK THEN

type t = Zip.out_file

BARISTA_ERROR =
  | Unable_to_open_stream of (path : Path.t) ->
      Printf.sprintf "unable to open archive output stream (%S)"
        (Path.to_string path)
  | Unable_to_write_data ->
      "unable to write data"
  | Unable_to_close_stream ->
      "unable to close archive stream"

external string_of_path : Path.t -> string =
  "%identity"

external string_of_bytes : Bytes.t -> string =
  "%identity"

let make_of_path ?(comment = @"") path =
  try
    path
    |> string_of_path
    |> Zip.open_out ~comment:(UTF8.to_string comment)
  with _ ->
    fail (Unable_to_open_stream path)

let add_entry archive ?(extra = Bytes.empty) ?(comment = @"") name data =
  try
    Zip.add_entry
      ~extra:(string_of_bytes extra)
      ~comment:(UTF8.to_string comment)
      (string_of_bytes data)
      archive
      (UTF8.to_string name)
  with _ ->
    fail Unable_to_write_data

let add_entry_from_file archive ?(extra = Bytes.empty) ?(comment = @"") name path =
  try
    Zip.copy_file_to_entry
      ~extra:(string_of_bytes extra)
      ~comment:(UTF8.to_string comment)
      (string_of_path path)
      archive
      (UTF8.to_string name)
  with _ ->
    fail Unable_to_write_data

let add_entry_from_stream archive ?(extra = Bytes.empty) ?(comment = @"") name stream =
let buffer_size = 65536 in
  let buffer = Bytes.make_of_size buffer_size in
  let data = ByteBuffer.make_of_size buffer_size in
  let last_read = ref buffer_size in
  while !last_read = buffer_size do
    last_read :=
      InputStream.read_available_bytes stream buffer_size buffer 0;
    if !last_read <> 0 then
      ByteBuffer.add_bytes_from data buffer 0 !last_read
  done;
  data
  |> ByteBuffer.contents
  |> add_entry archive ~extra ~comment name

let close archive =
  try
    Zip.close_out archive
  with _ ->
    fail Unable_to_close_stream

let close_noerr archive =
  try
    Zip.close_out archive
  with _ ->
    ()

ELSE (* USE_JDK *)

type t = java'util'zip'ZipOutputStream java_instance

BARISTA_ERROR =
  | Unable_to_open_stream of (path : Path.t) ->
      Printf.sprintf "unable to open archive output stream (%S)"
        (Path.to_string path)
  | Unable_to_write_data ->
      "unable to write data"
  | Unable_to_close_stream ->
      "unable to close archive stream"

external file_of_path : Path.t -> java'io'File java_instance =
  "%identity"

external java_string_of_utf8 : UTF8.t -> java'lang'String java_instance =
  "%identity"

external byte_array_of_bytes : Bytes.t -> int JavaByteArray.t =
  "%identity"

external data_input_stream_of_input_stream : InputStream.t -> java'io'DataInputStream java_instance =
  "%identity"

external byte_array_length : int java_byte_array -> int32 =
  "java array length byte"

let make_of_path ?(comment = @"") path =
  try
    let res =
      path
      |> file_of_path
      |> Java.make "java.io.FileOutputStream(java.io.File)"
      |> Java.make "java.util.zip.ZipOutputStream(_)" in
    let comment = java_string_of_utf8 comment in
    if (Java.call "String.length()" comment) > 0l then
      Java.call "java.util.zip.ZipOutputStream.setComment(_)"
        res
        comment;
    res
  with Java_exception _ ->
    fail (Unable_to_open_stream path)

let add_entry archive ?(extra = Bytes.empty) ?(comment = @"") name data =
  try
    let entry =
      Java.make "java.util.zip.ZipEntry(String)"
        (java_string_of_utf8 name) in
    let extra = byte_array_of_bytes extra in
    if (byte_array_length extra) > 0l then
      Java.call "java.util.zip.ZipEntry.setExtra(_)"
        entry
        extra;
    let comment = java_string_of_utf8 comment in
    if (Java.call "String.length()" comment) > 0l then
      Java.call "java.util.zip.ZipEntry.setComment(_)"
        entry
        comment;
    Java.call "java.util.zip.ZipOutputStream.putNextEntry(_)"
      archive
      entry;
    Java.call "java.util.zip.ZipOutputStream.write(_,_,_)"
      archive
      (byte_array_of_bytes data)
      0l
      (JavaByteArray.length (byte_array_of_bytes data));
    Java.call "java.util.zip.ZipOutputStream.closeEntry()" archive
  with Java_exception _ ->
    fail Unable_to_write_data

let add_entry_from_stream archive ?(extra = Bytes.empty) ?(comment = @"") name stream =
  let buffer_size = 65536l in
  let buffer = Java.make_array "byte[]" buffer_size in
  let stream = data_input_stream_of_input_stream stream in
  try
    let entry =
      Java.make "java.util.zip.ZipEntry(String)"
        (java_string_of_utf8 name) in
    let extra = byte_array_of_bytes extra in
    if (byte_array_length extra) > 0l then
      Java.call "java.util.zip.ZipEntry.setExtra(_)"
        entry
        extra;
    let comment = java_string_of_utf8 comment in
    if (Java.call "String.length()" comment) > 0l then
      Java.call "java.util.zip.ZipEntry.setComment(_)"
        entry
        comment;
    Java.call "java.util.zip.ZipOutputStream.putNextEntry(_)"
      archive
      entry;
    let last_read = ref buffer_size in
    while !last_read = buffer_size do
      last_read := Java.call "java.io.DataInputStream.read(_)" stream buffer;
      if !last_read <> -1l then
        Java.call "java.util.zip.ZipOutputStream.write(_,_,_)"
          archive
          buffer
          0l
          !last_read
    done;
    Java.call "java.util.zip.ZipOutputStream.closeEntry()" archive
  with Java_exception _ ->
    fail Unable_to_write_data

let add_entry_from_file archive ?(extra = Bytes.empty) ?(comment = @"") name path =
  path
  |> InputStream.make_of_path
  |> add_entry_from_stream archive ~extra ~comment name

let close archive =
  try
    Java.call "java.util.zip.ZipOutputStream.close()" archive
  with Java_exception _ ->
    fail Unable_to_close_stream

let close_noerr archive =
  try
    Java.call "java.util.zip.ZipOutputStream.close()" archive
  with Java_exception _ ->
    ()

END
