(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Creation of archives (zip, jar, etc.).

    This definition is independent of the actual implementation, that can
    be based on either the OCaml standard library, or the OCaml-Java
    runtime library. *)


type t
(** The type of archives. *)

BARISTA_ERROR =
  | Unable_to_open_stream of Path.t
  | Unable_to_write_data
  | Unable_to_close_stream

val make_of_path : ?comment:UTF8.t -> Path.t -> t
(** Creates a new archive with the passed path, and optional
    comment (defaulting to the empty string).

    Raises [Exception] if the archive cannot be created. *)

val add_entry : t -> ?extra:Bytes.t -> ?comment:UTF8.t -> UTF8.t -> Bytes.t -> unit
(** [add_entry archive ~extra ~comment name bytes] adds an entry named
    [name] to [archive], with data [bytes] and optional comment and extra
    data (both defaulting to empty values).

    Raises [Exception] if data cannot be written. *)

val add_entry_from_file : t -> ?extra:Bytes.t -> ?comment:UTF8.t -> UTF8.t -> Path.t -> unit
(** [add_entry_from_file archive ~extra ~comment name path] adds an entry
    named [name] to [archive], with data read from file [path] and
    optional comment and extra data (both defaulting to empty values).

    Raises [Exception] if data cannot be written.

    Raises [InputStream.Exception] if data cannot be read from file. *)

val add_entry_from_stream : t -> ?extra:Bytes.t -> ?comment:UTF8.t -> UTF8.t -> InputStream.t -> unit
(** [add_entry_from_stream archive ~extra ~comment name stream] adds an
    entry named [name] to [archive], with data read from [stream] and
    optional comment and extra data (both defaulting to empty values).

    Raises [Exception] if data cannot be written.

    Raises [InputStream.Exception] if data cannot be read from stream. *)

val close : t -> unit
(** Closes the passed archive.

    Raises [Exception] if an error occurs while closing the archive. *)

val close_noerr : t -> unit
(** Same as [close] but raised exceptions are silently discarded. *)
