(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

exception Bad_element of string

type string_switch = {
    switch_desc : UTF8.t;
    handle_string : UTF8.t -> unit;
  }

type choice_switch = {
    switch_options : UTF8.t list;
    handle_choice : UTF8.t -> unit;
  }

type kind =
  | Nothing of (unit -> unit)
  | String of string_switch
  | Choice of choice_switch

type switch = {
    identifier : UTF8.t;
    kind : kind;
    documentation : UTF8.t;
  }

exception Bad_argument of string

type others = {
    argument_desc : UTF8.t;
    handle_argument : (UTF8.t -> unit)
  }

let parse switches others args idx =
  let curr_idx = ref idx in
  let len_args = Array.length args in
  while !curr_idx < len_args do
    let curr_arg = args.(!curr_idx) in
    incr curr_idx;
    let switch =
      try
        Some (List.find
                (fun sw -> UTF8.equal curr_arg sw.identifier)
                switches)
      with Not_found ->
        None in
    match switch with
    | Some { identifier; kind; _ } ->
        (match kind with
        | Nothing func ->
            (try
              func ()
            with Bad_element error_message ->
              Printf.eprintf "*** %s\n%!" error_message;
              exit 1)
          | String { switch_desc; handle_string } ->
              if !curr_idx < len_args then begin
                let curr_elem = args.(!curr_idx) in
                incr curr_idx;
                try
                  handle_string curr_elem
                with Bad_element error_message ->
                  Printf.eprintf "*** wrong element %S passed to switch %S:\n%s%!"
                    (UTF8.to_string_noerr curr_elem)
                    (UTF8.to_string_noerr identifier)
                    error_message;
                  exit 1
              end else begin
                Printf.eprintf "*** switch %S is expecting one %s\n%!"
                  (UTF8.to_string_noerr identifier)
                  (UTF8.to_string_noerr switch_desc);
                exit 1
              end
          | Choice { switch_options; handle_choice } ->
              if !curr_idx < len_args then begin
                let curr_elem = args.(!curr_idx) in
                incr curr_idx;
                let valid =
                  List.exists
                    (fun opt -> UTF8.equal opt curr_elem)
                    switch_options in
                if valid then begin
                  try
                    handle_choice curr_elem
                  with Bad_element error_message ->
                    Printf.eprintf "*** wrong element passed to switch %S:\n%s\n%!"
                      (UTF8.to_string_noerr identifier)
                      error_message;
                    exit 1
                end else begin
                  let values = UTF8.concat_sep @", " switch_options in
                  Printf.eprintf "*** switch %S is expecting a value among %s\n%!"
                    (UTF8.to_string_noerr identifier)
                    (UTF8.to_string_noerr values);
                  exit 1
                end
              end else begin
                let values = UTF8.concat_sep @", " switch_options in
                Printf.eprintf "*** switch %S is expecting a value among %s\n%!"
                  (UTF8.to_string_noerr identifier)
                  (UTF8.to_string_noerr values);
                exit 1
              end)
    | None ->
        (match others with
        | Some { argument_desc; handle_argument } ->
            (try
              handle_argument curr_arg
            with Bad_argument error_message ->
              Printf.eprintf "*** wrong argument %S, expecting %s:\n%s\n%!"
                (UTF8.to_string_noerr curr_arg)
                (UTF8.to_string_noerr argument_desc)
                error_message;
              exit 1)
        | None ->
            Printf.eprintf "*** unexpected argument %S%!"
              (UTF8.to_string_noerr curr_arg);
            exit 1)
  done
