(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

IFNDEF USE_JDK THEN

type t = CamomileLibrary.UChar.t

BARISTA_ERROR =
  | Unrepresentable_character of (ch : t) ->
      Printf.sprintf "unrepresentable character (code '%d')"
        (CamomileLibrary.UChar.uint_code ch)
  | Invalid_character_code of (code : int) ->
      Printf.sprintf "invalid character code '%d'" code

let of_char ch =
  CamomileLibrary.UChar.of_char ch

let to_char ch =
  try
    CamomileLibrary.UChar.char_of ch
  with CamomileLibrary.UChar.Out_of_range ->
    fail (Unrepresentable_character ch)

let to_char_noerr ch =
  try
    CamomileLibrary.UChar.char_of ch
  with _ ->
    '?'

let of_code code =
  try
    CamomileLibrary.UChar.chr code
  with Invalid_argument _ ->
    fail (Invalid_character_code code)

let to_code ch =
  try
    CamomileLibrary.UChar.code ch
  with CamomileLibrary.UChar.Out_of_range ->
    fail (Unrepresentable_character ch)

let equal ch1 ch2 =
  CamomileLibrary.UChar.eq ch1 ch2

let compare ch1 ch2 =
  CamomileLibrary.UChar.compare ch1 ch2

let hash ch =
  CamomileLibrary.UChar.code ch

module CharInfo =
  CamomileLibrary.UCharInfo.Make (CamomileLibraryDefault.Config)

let is_letter ch =
  match CharInfo.general_category ch with
  | `Lu | `Ll | `Lt | `Lo | `Lm -> true
  | _ -> false

let is_digit ch =
  match CharInfo.general_category ch with
  | `Nd | `Nl | `No -> true
  | _ -> false

let underscore = of_char '_'

let is_identifier_start ch =
  equal underscore ch
  || is_letter ch

let is_identifier_part ch =
  equal underscore ch
  || is_letter ch
  || is_digit ch

let space = CamomileLibrary.UChar.of_char ' '

let tab = CamomileLibrary.UChar.of_char '\t'

let is_whitespace ch =
  (CamomileLibrary.UChar.eq ch space)
  || (CamomileLibrary.UChar.eq ch tab)

ELSE (* USE_JDK *)

let (|>) x f = f x

type t = int32 (* actually a code point *)

BARISTA_ERROR =
  | Unrepresentable_character of (ch : t) ->
      Printf.sprintf "unrepresentable character (code '%ld')" ch
  | Invalid_character_code of (code : int) ->
      Printf.sprintf "invalid character code '%d'" code

external of_char : char -> t =
  "%int32_of_int"

external unsafe_chr : int -> char =
  "%identity"

let to_char ch =
  if (ch >= 0l) && (ch < 0x100l) then
    unsafe_chr (Int32.to_int ch)
  else
    fail (Unrepresentable_character ch)

let to_char_noerr ch =
  if (ch >= 0l) && (ch < 0x100l) then
    unsafe_chr (Int32.to_int ch)
  else
    '?'

let of_code code =
  if (code >= 0) && (code lsr 31 = 0) then
    Int32.of_int code
  else
    fail (Invalid_character_code code)

let to_code ch =
  if (ch >= 0l) then
    Int32.to_int ch
  else
    fail (Unrepresentable_character ch)

let equal (ch1 : t) (ch2 : t) =
  ch1 = ch2

let compare ch1 ch2 =
  let cmp = Int32.(sub
                     (shift_right_logical ch1 16)
                     (shift_right_logical ch2 16)) in
  (if cmp = 0l then
      Int32.(sub (logand ch1 0xFFFFl) (logand ch2 0xFFFFl))
    else
      cmp)
  |> Int32.to_int

let hash ch =
  Int32.to_int ch

let is_letter ch =
  Java.call "Character.isLetter(int)" ch

let is_digit ch =
  Java.call "Character.isDigit(int)" ch

let is_identifier_start ch =
  Java.call "Character.isJavaIdentifierStart(int)" ch

let is_identifier_part ch =
  Java.call "Character.isJavaIdentifierPart(int)" ch

let is_whitespace ch =
  Java.call "Character.isWhitespace(int)" ch

END
