/*
 * JavaTryCmd.java --
 *
 *	Implements the built-in "java::try" command.
 *
 * Copyright (c) 1998 by Moses DeJong
 *
 * See the file "license.terms" for information on usage and
 * redistribution of this file, and for a DISCLAIMER OF ALL
 * WARRANTIES.
 *
 * RCS: @(#) $Id: JavaTryCmd.java,v 1.6 2006/04/27 02:16:13 mdejong Exp $
 *
 */

package tcl.lang;

/**
 * This class implements the built-in "java::try" command. The command
 * provides access to the java's try-catch-finally construct.
 */

public class JavaTryCmd implements Command
{

/**

 *----------------------------------------------------------------------
 *
 * cmdProc --
 *
 *      This procedure is invoked as part of the Command interface to
 *      process the "java::try" Tcl command.  See the user documentation
 *      for details on what it does.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

    public void cmdProc(Interp interp, TclObject[] argv)
	throws TclException {

	// (argv.length % 3) == 1 is a try+catch+finally block
	// (argv.length % 3) == 2 is a try+catch block

	int argv_length_mod = argv.length % 3;
	boolean try_catch_finally = false;
	boolean try_catch = false;

	if (argv_length_mod == 1) {
	    try_catch_finally = true;
	} else if (argv_length_mod == 2) {
	    try_catch = true;
	}

	if (argv.length < 4 || (!try_catch && !try_catch_finally)) {
	    throw new TclNumArgsException(interp,1,argv,
		      "script ?catch exception_pair script? ?finally script?");
	}

	// Eval the script argument
        eval(interp, argv[1]);

        // If an exception was raised while executing the script
        // we need to iterate over the catch blocks to see if
        // there is an exception listed that matches the one we
        // caught. Check for errors in the format of the catch
        // arguments as we go.

	if (exrec.exception_thrown) {
	    if (debug) {
	        System.out.print("Exception thrown inside body block, type = ");
	        if (exrec.reflect_exception != null)
	            System.out.println("ReflectException");
	        else if (exrec.tcl_exception != null)
	            System.out.println("TclException");
	        else if (exrec.runtime_exception != null)
	            System.out.println("RuntimeException");
	        else
	            throw new TclRuntimeError("Java exception not found");
	    }

	    int end_loop = argv.length;

	    // If there is a finally block then do not check it
	    // in this catch block processing loop
	    if (try_catch_finally) {
		end_loop -= 2;
	    }

	    for (int i=2; i < end_loop; i+=3) {
		TclObject catch_clause = argv[i];
		TclObject exception_pair = argv[i+1];
		TclObject exception_script = argv[i+2];

		if (! catch_clause.toString().equals("catch")) {
		    throw new TclException(interp,"invalid catch clause \"" +
					   catch_clause.toString() + "\"");
		}

		boolean exception_pair_problem = false;
		TclObject type = null;
		TclObject var = null;

		try {
		    if (TclList.getLength(interp,exception_pair) != 2) {
			exception_pair_problem = true;
		    } else {
			type =  TclList.index(interp,exception_pair,0);
			var  =  TclList.index(interp,exception_pair,1);

			// double check to make sure that these TclObjects
			// are not lists (they can only have one element)

			if (TclList.getLength(interp,type) != 1 ||
			    TclList.getLength(interp,var) != 1) {
			    exception_pair_problem = true;
			}
		    }
		} catch (TclException e) {
		    exception_pair_problem = true;
		}
		

		// if we could not parse the exception pair into a two
		// element list then we raise an error

		if (exception_pair_problem) {
		    throw new TclException(interp,"invalid exception_pair \"" +
					   exception_pair.toString() + "\"");
		}


	        if (debug) {
		System.out.println("type is \"" + type.toString() + "\"");
		System.out.println("var is \"" + var.toString() + "\"");
		System.out.println("script is \"" + exception_script.toString() + "\"");
	        }


		// now we need to "match" the name of the exception
		// to the name of the "type" of the exception_pair
		// we first try to match the long form of the name
		// (ie "java.lang.Exception" matches "java.lang.Exception")
		// but if that does not work then we try the short form
		// (ie "java.lang.Exception" matches "Exception")
		// We also watch for the special case where the identifier
		// is "TclException". In this case we match any exception
		// generated by a Tcl command. Any exception derived
		// from TclException will also be matched by this case

		
		// this will hold the Class that was matched
		// after the search is done
		Class matched_class = null;

		String type_name = type.toString();
		int    type_len  = type_name.length();

	        if (debug) {
		System.out.println("type_name is \"" + type_name + "\"");
	        }

		Class ex_type;
		if (exrec.reflect_exception != null) {
		    ex_type = exrec.reflect_exception.getThrowable().getClass();
		} else if (exrec.tcl_exception != null) {
		    ex_type = exrec.tcl_exception.getClass();
		} else if (exrec.runtime_exception != null) {
		    ex_type = exrec.runtime_exception.getClass();
		} else {
		    throw new TclRuntimeError("Exception not found");
		}
                
                if (debug) {
                    System.out.println("ex_type.getName() is " + ex_type.getName());
                }

		String ex_type_name;

		// Check for special case of TclInterruptedException.
		// This runtime exception is defined only in Jacl.
		// This exception can't be caught since it is
		// raised to unwind the stack and terminate execution.

		if (type_name.equals("TclInterruptedException") ||
		        type_name.equals("tcl.lang.TclInterruptedException")) {
                    throw new TclException(interp,
                        "can't catch TclInterruptedException");
                } else if ((exrec.runtime_exception != null) &&
                            ex_type.getName().equals(
                                "tcl.lang.TclInterruptedException")) {

	            // Scripts can't be allowed to catch TclInterruptedException
                    // since this is raised when interp is being taken down.

	            if (debug) {
		        System.out.println("skipped TclInterruptedException");
	            }

	            break; // out of for loop

		// check for special case where the type name is
		// "TclException" and we caught a TclException

		} else if (type_name.equals("TclException") ||
		    (exrec.tcl_exception != null)) {

		    if (type_name.equals("TclException") &&
		        (exrec.tcl_exception != null)) {
		        matched_class = TclException.class;

	                if (debug) {
			System.out.println("match for \"" +
					   matched_class.getName() +
					   "\" == \"" + type_name + "\"");
	                }
		    } else {
		        // If the exception name is "TclException" but the
		        // exception is not derived from TclException or
		        // if the type is derived from TclException but
		        // the type name is not "TclException", then 
		        // move on to the next catch block

	                if (debug) {
			System.out.println("skipping catch block because " +
			    "of TclException mismatch");
			System.out.println("exception name = \""
                            + type_name + "\"");
			System.out.println("caught TclException = " +
		            (exrec.tcl_exception != null));
	                }

		        continue; //for loop
		    }
		} else {

		  while (ex_type != null) {
		      ex_type_name = ex_type.getName();

	              if (debug) {
		      System.out.println("ex_type_name is \"" + ex_type_name + "\"");
	              }

		      if (ex_type_name.equals( type_name )) {
			  matched_class = ex_type; //full name match
		      } else {
			  // try to match based on the "short" name of the class
			  // so "Exception" would match "java.lang.Exception"
			  // Watch out!
			  // so "Exception" does not match "java.lang.Exception2"
		    
			  int last = ex_type_name.lastIndexOf('.');

			  /*
	                  if (debug) {
			  System.out.println("last is " + last);
			  System.out.println((last != -1) + " && " +
				((ex_type_name.length() - (last+1)) == type_len));

			  System.out.println("regionmatch is " +
			      ex_type_name.regionMatches(last+1,type_name,
							   0,type_len));
	                  }
			  */

			  if ((last != -1) &&
			      ((ex_type_name.length() - (last+1)) == type_len) &&
			      ex_type_name.regionMatches(last+1,type_name,0,type_len)) {

			      matched_class = ex_type;
			  }
		      }


		    if (matched_class != null) {
	                if (debug) {
			System.out.println("match for \"" +
					   matched_class.getName() +
					   "\" == \"" + type_name + "\"");
	                }
			break; // end this while loop when match is found
		    }



		      // java.lang.Throwable is the highest a catch can go
		      if (ex_type == Throwable.class) {
			  ex_type = null;
		      } else {
			  ex_type = ex_type.getSuperclass();
		      }

		  } // end while loop

                } // end else




		// if we found a match on this catch block then eval that code
		if (matched_class != null) {

		    // Set the value of the variable named in the exception_pair
		    // I think it would be best to just write over the value
		    // of the exception variable. We really do not want to
		    // quit if it already defined and addind a "scope" to
		    // an exception handler seems to go against Tcl constructs

		    // Case 1: The exception type matched is a TclException
		    // In this case the value of the exception variable is
		    // the error result of the command.

		    if (matched_class == TclException.class) {

	              if (debug) {
		      System.out.println("TclException result getting saved in exception varaible");
	              }

		      TclObject res = interp.getResult();
                      res.preserve();

		      try {
			interp.setVar(var.toString(), res, 0);
		      } catch (TclException e) {
			//this should never happen
			throw new TclRuntimeError(
			  "could not reflect or set exception variable");
		      }

                      res.release();

		    } else {

		    // Case 2: The exception type matched is a Java exception
		    // that is not a subclass of TclException. In this case
		    // the value of the exception variable is the exception
		    // object reflection handle (java0x1 or something).

	              if (debug) {
		      System.out.println("JavaException result getting saved in exception varaible");
	              }

                      Throwable t;

		      if (exrec.reflect_exception != null) {
		          t = exrec.reflect_exception.getThrowable();
		      } else if (exrec.runtime_exception != null) {
		          t = exrec.runtime_exception;
		      } else {
                          throw new TclRuntimeError("Java exception not found");
		      }

		      try {
			interp.setVar(var.toString(),
			  ReflectObject.newInstance(interp,
			    matched_class,t)
                          ,0);
		      } catch (TclException e) {
			// this should never happen
			throw new TclRuntimeError(
			  "could not reflect or set exception variable");
		      }

		    }

		    // Now eval the exception handler script
		    // this will also reset exrec values.
		    eval(interp, exception_script);

	            if (debug) {
		        if (exrec.exception_thrown)
		            System.out.println("exception thrown by exception handler");
		        else
		            System.out.println("no exception thrown by exception handler");
	            }

		    break; // break out of the enclosing for loop
		           // this will stop catch block processing
		}

	    } // end for loop

	} // end if block




	// check and run the finally block if there is one

	if (try_catch_finally) {
	    
	    TclObject finally_clause = argv[argv.length - 2];
	    TclObject finally_script = argv[argv.length - 1];

	    if (! finally_clause.toString().equals("finally")) {
		throw new TclException(interp,"invalid finally clause \"" +
				       finally_clause.toString() + "\"");
	    }


	    if (debug) {
	    System.out.println("finally script is \"" +
			       finally_script.toString() + "\"");
	    }


	    // the finally script might change the interpreter result
            // so we need to save the current result so that it can
	    // restored after running the finally script

	    TclObject res = interp.getResult();
	    res.preserve();
	    interp.resetResult();

	    // evaluate the finally scipt and make sure that exceptions
	    // in the finally script are caught and then thrown at
	    // the very end of this method. Be sure to save the
	    // exception record value from the catch block evaluation
	    // so we can rethrow any raised exception if the catch
	    // block does not raise any exceptions.

	    ExRecord tmp = exrec;
	    exrec = tmp_exrec;
	    tmp_exrec = tmp;

	    eval(interp, finally_script);

	    // If the finally block did not raise an error
	    // then reset the previous interpreter result
	    // and use the previous exception record.
            // Also reset when the original exception was
            // a TclInterruptedException since an exception
            // in the finally block is ignored in that case.

	    if ((exrec.exception_thrown == false) ||
                    ((exrec.exception_thrown == true) &&
                        (tmp_exrec.runtime_exception != null) &&
                        tmp_exrec.runtime_exception.getClass().getName().equals(
                            "tcl.lang.TclInterruptedException"))) {

	        if (debug) {
	            System.out.println("resetting result and exception record");
	        }

	        interp.setResult(res);
	        tmp = exrec;
	        exrec = tmp_exrec;
	        tmp_exrec = tmp;
            }

	    res.release();
	}


	// If an exception was thrown in the catch block or the
	// finally block, then we need to throw it again.

	if (exrec.exception_thrown) {
	    if (debug) {
	        System.out.print("throwing end of try method exception, type = ");
	        if (exrec.reflect_exception != null)
	            System.out.println("ReflectException");
	        else if (exrec.tcl_exception != null)
	            System.out.println("TclException");
	        else if (exrec.runtime_exception != null)
	            System.out.println("RuntimeException");
	        else
	            throw new TclRuntimeError("Java exception not found");
	    }

	    if (exrec.reflect_exception != null) {
	        if (debug) {
		    System.out.println("rethrowing ReflectException " +
		            exrec.reflect_exception);
	        }

		throw exrec.reflect_exception;
	    } else if (exrec.tcl_exception != null) {
	        if (debug) {
		    System.out.println("rethrowing TclException " +
		            exrec.tcl_exception);
	        }

		throw exrec.tcl_exception;
	    } else if (exrec.runtime_exception != null) {
	        if (debug) {
		    System.out.println("rethrowing RuntimeException " +
		            exrec.runtime_exception);
	        }

		throw exrec.runtime_exception;
	    } else {
	        throw new TclRuntimeError("Java exception not found");
	    }
	}
    }

    private static class ExRecord {
        // Wrapper type, holds real exception thrown by some invoked method
        ReflectException reflect_exception;

        // An actual TclException
        TclException tcl_exception;

        // Some undeclared exception that was not raised in an invoked method.
        RuntimeException runtime_exception;

        // True if one of the three exceptions types was caught
        boolean exception_thrown;
    }

    ExRecord exrec = new ExRecord();
    ExRecord tmp_exrec = new ExRecord();

    private void eval(Interp interp, TclObject script) {
        exrec.reflect_exception = null;
        exrec.tcl_exception = null;
        exrec.runtime_exception = null;
        exrec.exception_thrown = false;

        try {
            interp.eval(script, 0);
        } catch (ReflectException e) {
            if (debug) {
	        System.out.println("eval() caught ReflectException");
	    }
            exrec.reflect_exception = e;
            exrec.exception_thrown = true;
	} catch (TclException e) {
	    if (debug) {
	        System.out.println("eval() caught TclException");
	    }
            exrec.tcl_exception = e;
            exrec.exception_thrown = true;

            // Exception might have been return, break, or continue
            int code = e.getCompletionCode();

	    if (code == TCL.RETURN) {
	        // If the caught exception is a regular return
	        // then no error really occured and we ignore it
	        exrec.exception_thrown = false;
	    }
            // Process TCL.ERROR, TCL.BREAK, or TCL.CONTINUE
            // like any other exception
        } catch (RuntimeException e) {
	    if (debug) {
	        System.out.println("catching RuntimeException");
	    }

            exrec.runtime_exception = e;
            exrec.exception_thrown = true;
        }
    }

    private static final boolean debug = false; // enables debug output

} // end of JavaTryCmd class
