(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Class types and default implementations for "mapper", "iterator", and
    "folder" over archives. *)

type archive_entry =
  | Directory_entry of ArchiveEntry.t (** Simple entry, with no data. *)
  | Package_entry of ArchiveEntry.t * Bytes.t (** Package entry, along with data. *)
  | Class_entry of ArchiveEntry.t * Bytes.t (** Class entry, along with data. *)
  | Bare_entry of ArchiveEntry.t * Bytes.t (** Bare entry, along with data. *)
(** The type of archive entries (either simple entry, or with
    associated data as bytes). *)

class type archive_mapper = object
  method comment : UTF8.t -> UTF8.t
  method entry : ArchiveEntry.t -> archive_entry
  method directory_entry : ArchiveEntry.t -> ArchiveEntry.t
  method package_entry : ArchiveEntry.t -> ArchiveEntry.t
  method class_entry : ArchiveEntry.t -> ArchiveEntry.t
  method bare_entry : ArchiveEntry.t -> ArchiveEntry.t
  method package_definition : PackageDefinition.t -> PackageDefinition.t
  method class_definition : ClassDefinition.t -> ClassDefinition.t
  method data : Bytes.t -> Bytes.t
end
(** This class type defines a "mapper", instances being used as
    "functions" to transform archives. The "function" is defined {i by
    parts} through the various methods of the object. Any method is
    responsible for the calling of its embedded elements. *)

class default_archive_mapper : ArchiveFile.t -> archive_mapper
(** The default "mapper", that encodes the identity over archives. *)

class type archive_iterator = object
  method comment : UTF8.t -> unit
  method entry : ArchiveEntry.t -> unit
  method directory_entry : ArchiveEntry.t -> unit
  method package_entry : ArchiveEntry.t -> unit
  method class_entry : ArchiveEntry.t -> unit
  method bare_entry : ArchiveEntry.t -> unit
  method package_definition : PackageDefinition.t -> unit
  method class_definition : ClassDefinition.t -> unit
  method data : Bytes.t -> unit
end
(** This class type defines an "iterator", instances being used as
    "functions" to transform archives. The "function" is defined {i by
    parts} through the various methods of the object. Any method is
    responsible for the calling of its embedded elements. *)

class default_archive_iterator : ArchiveFile.t -> archive_iterator
(** The default "iterator", that does nothing. *)

class type archive_folder = object ('self)
  method comment : UTF8.t -> 'self
  method entry : ArchiveEntry.t -> 'self
  method directory_entry : ArchiveEntry.t -> 'self
  method package_entry : ArchiveEntry.t -> 'self
  method class_entry : ArchiveEntry.t -> 'self
  method bare_entry : ArchiveEntry.t -> 'self
  method package_definition : PackageDefinition.t -> 'self
  method class_definition : ClassDefinition.t -> 'self
  method data : Bytes.t -> 'self
end
(** This class type defines a "folder", instances being used as
    "functions" to transform archives. The "function" is defined {i by
    parts} through the various methods of the object. Any method is
    responsible for the calling of its embedded elements. *)

class default_archive_folder : ArchiveFile.t -> archive_folder
(** The default "folder", that encodes the identity over archives. *)

val map : archive_mapper -> ArchiveFile.t -> Path.t -> ArchiveOutputStream.t
(** Applies the "mapper" to the zip file, the passed string being the
    name of the output file to create. *)

val iter : archive_iterator -> ArchiveFile.t -> unit
(** Applies the "iterator" to the zip file. *)

val fold : (archive_folder as 'a) -> ArchiveFile.t -> 'a
(** Applies the "folder" to the zip file. *)

val map_file : (ArchiveFile.t -> archive_mapper) -> ?post:(ArchiveOutputStream.t -> unit) -> Path.t -> Path.t -> unit
(** [map_file m ?post:p i o] applies the "mapper" [m] to the zip file
    whose name is [i], producing zip file [o]. The [p] callback is called
    after mapping.*)

val iter_file : (ArchiveFile.t -> archive_iterator) -> Path.t -> unit
(** [iter_file i n] applies the "iterator" [i] to the zip file whose name
    is [n]. *)

val fold_file : (ArchiveFile.t -> (archive_folder as 'a)) -> Path.t -> 'a
(** [fold_file f n] applies the "folder" [f] to the zip file whose name
    is [n]. *)
