(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


(** Definitions and utility functions for partial evaluation. *)


(** {6 Types} *)

type value =
  | Unknown_value (** Unknown value of unknown type (that is: {i top}). *)
  | Unknown_integer_value (** Unknown integer value. *)
  | Known_integer_value of int32 (** Known integer value. *)
  | Unknown_float_value (** Unknown float value. *)
  | Known_float_value of float (** Known float value. *)
  | Unknown_long_value (** Unknown long value. *)
  | Known_long_value of int64 (** Known long value. *)
  | Unknown_double_value (** Unknown double value. *)
  | Known_double_value of float (** Known double value. *)
  | Null_value (** {i null} value (when type is unknown). *)
  | Object_value (** Unknown reference value. *)
(** The type of partially evaluated values. *)

type locals
(** The type of locals: integer-index value information, elements being
    of [value] type. *)

type stack
(** The type of {i operand} stacks, elements being of [value] type. *)

type t = {
    locals : locals; (** type information for the locals. *)
    stack : stack; (** type information for the {i operand} stack. *)
  }
(** The type of evalutions. *)

type operation =
  | Pop (** Execute the {i pop} instruction. *)
  | Pop2 (** Execute the {i pop2} instruction. *)
  | Push_integer of int32 * (Utils.u2 option) (** Push the integer value onto the stack, keeping the load index if any. *)
  | Push_float of float * (Utils.u2 option) (** Push the float value onto the stack, keeping the load index if any. *)
  | Push_long of int64 * (Utils.u2 option) (** Push the long value onto the stack, keeping the load index if any. *)
  | Push_double of float * (Utils.u2 option) (** Push the double value onto the stack, keeping the load index if any. *)
  | Push_null of Utils.u2 option (** Push the {i null} value onto the stack, keeping the load index if any. *)
  | Execute of Instruction.t (** Just execute the instruction. *)
  | Sequence of operation list (** Execute the sequence of operations. *)
(** The type of operations, that are simplifications of instruction lists. *)

val map_operations : (Utils.u2 * operation) list -> (Utils.u2 * Instruction.t) list
(** Compiles the passed list of offset/operation couples into a list of
    offset/instruction couples. *)


(** {6 Exception} *)

BARISTA_ERROR =
  | Unsupported_instruction of string
  | Empty_stack
  | Invalid_local_index of Utils.u2 * int
  | Invalid_stack_top of Attribute.verification_type_info * Attribute.verification_type_info
  | Invalid_local_contents of Utils.u2 * Attribute.verification_type_info * Attribute.verification_type_info
  | Invalid_local_contents_placeholder of Utils.u2 * Attribute.verification_type_info
  | Reference_waited of Attribute.verification_type_info
  | Reference_waited_placeholder_found
  | Array_waited
  | Category1_waited
  | Category2_waited
  | Different_stack_sizes of int * int
  | Invalid_primitive_array_type


(** {6 Construction} *)

val make_empty : unit -> t
(** Returns an empty evaluation (no local, and no operand on stack). *)

val make_of_parameters : bool -> Descriptor.for_parameter list -> t
(** [make_of_parameters inst p] returns the evaluation describing the stack
    at the beginning of a method, [inst] indicating whether the method is
    an instance method., and [p] being the list of parameters of the method. *)

val make_of_method : Method.t -> t
(** [make_of_method m] returns the evaluation describing the stack at
    the beginning for the method [m] in the class [cn]. *)


(** {6 Access and modification} *)

val locals_size : t -> int
(** Returns the size of the locals for the passed evaluation. *)

val stack_size : t -> int
(** Returns the size of the operand stack for the passed evaluation. *)

val equal : t -> t -> bool
(** Equality over stack evaluations. *)

val push : value -> stack -> stack
(** [push v s] returns a stack similar to [s] with [v] pushed on its
    top. *)

val top : stack -> value
(** [top s] returns the top element of [s].
    Raises [Exception] if [s] is empty. *)

val pop : stack -> stack
(** [pop s] returns [s] without its top element.
    Raises [Exception] if [s] is empty. *)

val pop_if : value -> stack -> stack
(** [pop_if v s] returns [s] without its top element if this element is
    compatible with [v], raising [Exception] otherwise.
    Raises [Exception] if [s] is empty. *)

val pop_if_category1 : stack -> value * stack
(** [pop_if_category1 s] returns a couple with the top element, and [s]
    without its top element. Raises [Exception] if [s] is empty, or if
    its top element is not a {i category 1} element. *)

val empty : unit -> stack
(** Returns an empty stack. *)

val only_exception : unit -> stack
(** Returns a stack containing only the passed element. *)


(** {6 Operations} *)

val update : Instruction.t -> t -> t * operation
(** [update cn ofs i ev] returns the evaluation [ev] updated according to
    the instruction [i] located at offset [ofs] (in class [cn]). Also
    returns the operation that is equivalent to the passed instruction in
    the context of the passed evaluation.
    Raises [Exception] if the passed evaluation is not compatible with
    the instruction, or if the instruction is unsupported ({i jsr},
    {i jsr_w}, {i ret}, or {i wide ret}). *)

val unify : t -> t -> t
(** [unify u ev1 ev2] returns an evaluation that generalizes [ev1] and
    [ev2]. *)
