(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

open Consts


(* Types *)

type element =
  | Class of Utils.u2
  | Fieldref of Utils.u2 * Utils.u2
  | Methodref of Utils.u2 * Utils.u2
  | InterfaceMethodref of Utils.u2 * Utils.u2
  | String of Utils.u2
  | Integer of int32
  | Float of int32
  | Long of int32 * int32
  | Double of int32 * int32
  | NameAndType of Utils.u2 * Utils.u2
  | UTF8 of UTF8.t
  | MethodHandle of Reference.kind * Utils.u2
  | MethodType of Utils.u2
  | InvokeDynamic of Utils.u2 * Utils.u2
  | ModuleId of Utils.u2 * Utils.u2

let equal_element x y =
  match x, y with
  | (Class x1), (Class y1) -> x1 = y1
  | (Fieldref (x1, x2)), (Fieldref (y1, y2)) -> (x1 = y1) && (x2 = y2)
  | (Methodref (x1, x2)), (Methodref (y1, y2)) -> (x1 = y1) && (x2 = y2)
  | (InterfaceMethodref (x1, x2)), (InterfaceMethodref (y1, y2)) -> (x1 = y1) && (x2 = y2)
  | (String x1), (String y1) -> x1 = y1
  | (Integer x1), (Integer y1) -> x1 = y1 
  | (Float x1), (Float y1) -> x1 = y1
  | (Long (x1, x2)), (Long (y1, y2)) -> (x1 = y1) && (x2 = y2)
  | (Double (x1, x2)), (Double (y1, y2)) -> (x1 = y1) && (x2 = y2)
  | (NameAndType (x1, x2)), (NameAndType (y1, y2)) -> (x1 = y1) && (x2 = y2)
  | (UTF8 x1), (UTF8 y1) -> UTF8.equal x1 y1
  | (MethodHandle (x1, x2)), (MethodHandle (y1, y2)) -> (x1 = y1) && (x2 = y2)
  | (MethodType x1), (MethodType y1) -> x1 = y1
  | (InvokeDynamic (x1, x2)), (InvokeDynamic (y1, y2)) -> (x1 = y1) && (x2 = y2)
  | (ModuleId (x1, x2)), (ModuleId (y1, y2)) -> (x1 = y1) && (x2 = y2)
  | _ -> false

let compare_element x y =
  match x, y with
  | (Class x1), (Class y1) -> Pervasives.compare x1 y1
  | (Fieldref (x1, x2)), (Fieldref (y1, y2)) -> let res = Pervasives.compare x1 y1 in if res = 0 then Pervasives.compare x2 y2 else res
  | (Methodref (x1, x2)), (Methodref (y1, y2)) -> let res = Pervasives.compare x1 y1 in if res = 0 then Pervasives.compare x2 y2 else res
  | (InterfaceMethodref (x1, x2)), (InterfaceMethodref (y1, y2)) -> let res = Pervasives.compare x1 y1 in if res = 0 then Pervasives.compare x2 y2 else res
  | (String x1), (String y1) -> Pervasives.compare x1 y1
  | (Integer x1), (Integer y1) -> Pervasives.compare x1 y1 
  | (Float x1), (Float y1) -> Pervasives.compare x1 y1
  | (Long (x1, x2)), (Long (y1, y2)) -> let res = Pervasives.compare x1 y1 in if res = 0 then Pervasives.compare x2 y2 else res
  | (Double (x1, x2)), (Double (y1, y2)) -> let res = Pervasives.compare x1 y1 in if res = 0 then Pervasives.compare x2 y2 else res
  | (NameAndType (x1, x2)), (NameAndType (y1, y2)) -> let res = Pervasives.compare x1 y1 in if res = 0 then Pervasives.compare x2 y2 else res
  | (UTF8 x1), (UTF8 y1) -> UTF8.compare x1 y1
  | (MethodHandle (x1, x2)), (MethodHandle (y1, y2)) -> let res = Pervasives.compare x1 y1 in if res = 0 then Pervasives.compare x2 y2 else res
  | (MethodType x1), (MethodType y1) -> Pervasives.compare x1 y1
  | (InvokeDynamic (x1, x2)), (InvokeDynamic (y1, y2)) ->let res = Pervasives.compare x1 y1 in if res = 0 then Pervasives.compare x2 y2 else res
  | (ModuleId (x1, x2)), (ModuleId (y1, y2)) ->let res = Pervasives.compare x1 y1 in if res = 0 then Pervasives.compare x2 y2 else res
  | _ -> Pervasives.compare x y

let hash_element = function
  | Class x1 -> 1 + (Utils.universal_hash x1)
  | Fieldref (x1, x2) -> 2 + (Utils.universal_hash x1) + (Utils.universal_hash x2)
  | Methodref (x1, x2) -> 3 + (Utils.universal_hash x1) + (Utils.universal_hash x2)
  | InterfaceMethodref (x1, x2) -> 4 + (Utils.universal_hash x1) + (Utils.universal_hash x2)
  | String x1 -> 5 + (Utils.universal_hash x1)
  | Integer x1 -> 6 + (Utils.universal_hash x1)
  | Float x1 -> 7 + (Utils.universal_hash x1)
  | Long (x1, x2) -> 8 + (Utils.universal_hash x1) + (Utils.universal_hash x2)
  | Double (x1, x2) -> 9 + (Utils.universal_hash x1) + (Utils.universal_hash x2)
  | NameAndType (x1, x2) -> 10 + (Utils.universal_hash x1) + (Utils.universal_hash x2)
  | UTF8 x1 -> 11 + (UTF8.hash x1)
  | MethodHandle (x1, x2) -> 12 + (Utils.universal_hash x1) + (Utils.universal_hash x2)
  | MethodType x1 -> 13 + (Utils.universal_hash x1)
  | InvokeDynamic (x1, x2) -> 14 + (Utils.universal_hash x1) + (Utils.universal_hash x2)
  | ModuleId (x1, x2) -> 15 + (Utils.universal_hash x1) + (Utils.universal_hash x2)

let to_string_element = function
  | Class x -> Printf.sprintf "Class %d" (x :> int)
  | Fieldref (x, y) -> Printf.sprintf "Fieldref (%d, %d)" (x :> int) (y :> int)
  | Methodref (x, y) -> Printf.sprintf "Methodref (%d, %d)" (x :> int) (y :> int)
  | InterfaceMethodref (x, y) -> Printf.sprintf "InterfaceMethodref (%d, %d)" (x :> int) (y :> int)
  | String x -> Printf.sprintf "String %d" (x :> int)
  | Integer x -> Printf.sprintf "Integer %ld" x
  | Float x -> Printf.sprintf "Float %ld" x
  | Long (x, y) -> Printf.sprintf "Long (%ld, %ld)" x y
  | Double (x, y) -> Printf.sprintf "Double (%ld, %ld)" x y
  | NameAndType (x, y) -> Printf.sprintf "NameAndType (%d, %d)" (x :> int) (y :> int)
  | UTF8 x -> Printf.sprintf "UTF8 %S" (UTF8.to_string_noerr x)
  | MethodHandle (x, y) ->
      Printf.sprintf "MethodHandle (%s, %d)"
        (match x with
        | Reference.REF_getField -> "getField"
        | Reference.REF_getStatic -> "getStatic"
        | Reference.REF_putField -> "putField"
        | Reference.REF_putStatic -> "putStatic"
        | Reference.REF_invokeVirtual -> "invokeVirtual"
        | Reference.REF_invokeStatic -> "invokeStatic"
        | Reference.REF_invokeSpecial -> "invokeSpecial"
        | Reference.REF_newInvokeSpecial -> "newInvokeSpecial"
        | Reference.REF_invokeInterface -> "invokeInterface")
        (y :> int)
  | MethodType x -> Printf.sprintf "MethodType %d" (x :> int)
  | InvokeDynamic (x, y) -> Printf.sprintf "InvokeDynamic (%d, %d)" (x :> int) (y :> int)
  | ModuleId (x, y) -> Printf.sprintf "ModuleId (%d, %d)" (x :> int) (y :> int)

let dummy_element =
  UTF8 @"Dummy-Constant-Pool-Entry"

type tag =
  | CONSTANT_Class
  | CONSTANT_Fieldref
  | CONSTANT_Methodref
  | CONSTANT_InterfaceMethodref
  | CONSTANT_String
  | CONSTANT_Integer
  | CONSTANT_Float
  | CONSTANT_Long
  | CONSTANT_Double
  | CONSTANT_NameAndType
  | CONSTANT_Utf8
  | CONSTANT_MethodHandle
  | CONSTANT_MethodType
  | CONSTANT_InvokeDynamic
  | CONSTANT_ModuleId

type t = element array

let equal x y =
  Utils.array_equal equal_element x y

let compare x y =
  Utils.array_compare compare_element x y

let hash x =
  Utils.array_hash hash_element x


(* Exception *)

BARISTA_ERROR =
  | Invalid_tag of (x : Utils.u1) ->
      Printf.sprintf "invalid constant pool tag (%d)" (x :> int)
  | Too_large of (x : int) ->
      Printf.sprintf "constant pool is too large (%d)" x
  | Invalid_reference ->
      "invalid constant pool reference (0)"
  | Reference_out_of_bounds of (idx : int) * (sz : int) ->
      Printf.sprintf "constant pool reference out of bounds (index %d, length %d)" idx sz
  | Dummy_access of (x : Utils.u2) ->
      Printf.sprintf "access to dummy element (index %d)" (x :> int)
  | Malformed_Class_entry of (x : Utils.u2) ->
      Printf.sprintf "malformed Class entry (index %d)" (x :> int)
  | Malformed_Fieldref_entry of (x : Utils.u2) * (y : Utils.u2) ->
      Printf.sprintf "malformed Fieldref entry (indexes %d and %d)" (x :> int) (y :> int)
  | Malformed_Methodref_entry of (x : Utils.u2) * (y : Utils.u2) ->
      Printf.sprintf "malformed Methodref entry (indexes %d and %d)" (x :> int) (y :> int)
  | Malformed_InterfaceMethodRef_entry of (x : Utils.u2) * (y : Utils.u2) ->
      Printf.sprintf "malformed InterfaceMethodref entry (indexes %d and %d)" (x :> int) (y :> int)
  | Malformed_String_entry of (x : Utils.u2) ->
      Printf.sprintf "malformed String entry (index %d)" (x :> int)
  | Malformed_NameAndType_entry of (x : Utils.u2) * (y : Utils.u2) ->
      Printf.sprintf "malformed NameAndType entry (indexes %d and %d)" (x :> int) (y :> int)
  | Malformed_MethodHandle_entry of (x : Reference.kind) * (y : Utils.u2) ->
      Printf.sprintf "malformed MethodHandle entry (kind %d and index %d)" ((Reference.u1_of_kind x) :> int) (y :> int)
  | Malformed_MethodType_entry of (x : Utils.u2) ->
      Printf.sprintf "malformed MethodType entry (index %d)" (x :> int)
  | Malformed_InvokeDynamic_entry of (x : Utils.u2) * (y : Utils.u2) ->
      Printf.sprintf "malformed InvokeDynamic entry (indexes %d and %d)" (x :> int) (y :> int)
  | Malformed_ModuleId_entry of (x : Utils.u2) * (y : Utils.u2) ->
      Printf.sprintf "malformed ModuleId entry (indexes %d and %d)" (x :> int) (y :> int)


(* I/O functions *)

let element_size = function
  | Class _ -> 1
  | Fieldref _ -> 1
  | Methodref _ -> 1
  | InterfaceMethodref _ -> 1
  | String _ -> 1
  | Integer _ -> 1
  | Float _ -> 1
  | Long _ -> 2
  | Double _ -> 2
  | NameAndType _ -> 1
  | UTF8 _ -> 1
  | MethodHandle _ -> 1
  | MethodType _ -> 1
  | InvokeDynamic _ -> 1
  | ModuleId _ -> 1

let int_of_tag = function
  | CONSTANT_Class -> 7
  | CONSTANT_Fieldref -> 9
  | CONSTANT_Methodref -> 10
  | CONSTANT_InterfaceMethodref -> 11
  | CONSTANT_String -> 8
  | CONSTANT_Integer -> 3
  | CONSTANT_Float -> 4
  | CONSTANT_Long -> 5
  | CONSTANT_Double -> 6
  | CONSTANT_NameAndType -> 12
  | CONSTANT_Utf8 -> 1
  | CONSTANT_MethodHandle -> 15
  | CONSTANT_MethodType -> 16
  | CONSTANT_InvokeDynamic -> 18
  | CONSTANT_ModuleId -> 13

let u1_of_tag x =
  Utils.u1 (int_of_tag x)

let tag_of_int = function
  | 1 -> CONSTANT_Utf8
  | 3 -> CONSTANT_Integer
  | 4 -> CONSTANT_Float
  | 5 -> CONSTANT_Long
  | 6 -> CONSTANT_Double
  | 7 -> CONSTANT_Class
  | 8 -> CONSTANT_String
  | 9 -> CONSTANT_Fieldref
  | 10 -> CONSTANT_Methodref
  | 11 -> CONSTANT_InterfaceMethodref
  | 12 -> CONSTANT_NameAndType
  | 13 -> CONSTANT_ModuleId
  | 15 -> CONSTANT_MethodHandle
  | 16 -> CONSTANT_MethodType
  | 18 -> CONSTANT_InvokeDynamic
  | x -> fail (Invalid_tag (Utils.u1 x))

let read_element st =
  match tag_of_int ((InputStream.read_u1 st) :> int) with
  | CONSTANT_Utf8 ->
      UTF8 (InputStream.read_utf8 st)
  | CONSTANT_Integer ->
      let bytes = InputStream.read_s4 st in
      Integer (bytes :> int32)
  | CONSTANT_Float ->
      let bytes = InputStream.read_s4 st in
      Float (bytes :> int32)
  | CONSTANT_Long ->
      let high_bytes = InputStream.read_s4 st in
      let low_bytes = InputStream.read_s4 st in
      Long ((high_bytes :> int32), (low_bytes :> int32))
  | CONSTANT_Double ->
      let high_bytes = InputStream.read_s4 st in
      let low_bytes = InputStream.read_s4 st in
      Double ((high_bytes :> int32), (low_bytes :> int32))
  | CONSTANT_Class ->
      let name_index = InputStream.read_u2 st in
      Class name_index
  | CONSTANT_String ->
      let string_index = InputStream.read_u2 st in
      String string_index
  | CONSTANT_Fieldref ->
      let class_index = InputStream.read_u2 st in
      let name_and_type_index = InputStream.read_u2 st in
      Fieldref (class_index, name_and_type_index)
  | CONSTANT_Methodref ->
      let class_index = InputStream.read_u2 st in
      let name_and_type_index = InputStream.read_u2 st in
      Methodref (class_index, name_and_type_index)
  | CONSTANT_InterfaceMethodref ->
      let class_index = InputStream.read_u2 st in
      let name_and_type_index = InputStream.read_u2 st in
      InterfaceMethodref (class_index, name_and_type_index)
  | CONSTANT_NameAndType ->
      let name_index = InputStream.read_u2 st in
      let descriptor_index = InputStream.read_u2 st in
      NameAndType (name_index, descriptor_index)
  | CONSTANT_MethodHandle ->
      let reference_kind = InputStream.read_u1 st in
      let reference_index = InputStream.read_u2 st in
      MethodHandle (Reference.kind_of_u1 reference_kind, reference_index)
  | CONSTANT_MethodType ->
      let descriptor_index = InputStream.read_u2 st in
      MethodType descriptor_index
  | CONSTANT_InvokeDynamic ->
      let bootstrap_method_attr_index = InputStream.read_u2 st in
      let name_and_type_index = InputStream.read_u2 st in
      InvokeDynamic (bootstrap_method_attr_index, name_and_type_index)
  | CONSTANT_ModuleId ->
      let name_index = InputStream.read_u2 st in
      let version_index = InputStream.read_u2 st in
      ModuleId (name_index, version_index)

let read st sz =
  let sz = (sz : Utils.u2 :> int) in
  if (sz >= 0) && (sz <= Utils.max_u2_value) then
    let res = Array.make sz dummy_element in
    let i = ref 1 in
    while !i < sz do
      let e = read_element st in
      res.(!i) <- e;
      i := !i + (element_size e)
    done;
    res
  else
    fail (Too_large sz)

let write_element st e =
  if e != dummy_element then
    match e with
    | Class x ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_Class);
        OutputStream.write_u2 st x
    | Fieldref (x, y) ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_Fieldref);
        OutputStream.write_u2 st x;
        OutputStream.write_u2 st y
    | Methodref (x, y) ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_Methodref);
        OutputStream.write_u2 st x;
        OutputStream.write_u2 st y
    | InterfaceMethodref (x, y) ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_InterfaceMethodref);
        OutputStream.write_u2 st x;
        OutputStream.write_u2 st y
    | String x ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_String);
        OutputStream.write_u2 st x
    | Integer x ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_Integer);
        OutputStream.write_s4 st (Utils.s4 x)
    | Float x ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_Float);
        OutputStream.write_s4 st (Utils.s4 x)
    | Long (x, y) ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_Long);
        OutputStream.write_s4 st (Utils.s4 x);
        OutputStream.write_s4 st (Utils.s4 y)
    | Double (x, y) ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_Double);
        OutputStream.write_s4 st (Utils.s4 x);
        OutputStream.write_s4 st (Utils.s4 y)
    | NameAndType (x, y) ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_NameAndType);
        OutputStream.write_u2 st x;
        OutputStream.write_u2 st y
    | UTF8 x ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_Utf8);
        OutputStream.write_utf8 st x
    | MethodHandle (x, y) ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_MethodHandle);
        OutputStream.write_u1 st (Reference.u1_of_kind x);
        OutputStream.write_u2 st y
    | MethodType x ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_MethodType);
        OutputStream.write_u2 st x
    | InvokeDynamic (x, y) ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_InvokeDynamic);
        OutputStream.write_u2 st x;
        OutputStream.write_u2 st y
    | ModuleId (x, y) ->
        OutputStream.write_u1 st (u1_of_tag CONSTANT_ModuleId);
        OutputStream.write_u2 st x;
        OutputStream.write_u2 st y

let write st pool =
  let len = Array.length pool in
  if len <= Utils.max_u2_value then
    Array.iter (write_element st) pool
  else
    fail (Too_large len)


(* Check functions *)

let size pool =
  Utils.u2 (Array.length pool)

let get_entry pool i =
  let i = (i : Utils.u2 :> int) in
  if i = 0 then
    fail Invalid_reference
  else if (i < 0) || (i >= (Array.length pool)) then
    fail (Reference_out_of_bounds (i, Array.length pool))
  else
    let res = pool.(i) in
    if res == dummy_element then
      fail (Dummy_access (Utils.u2 i))
    else
      res

let check pool =
  let get_entry = get_entry pool in
  let check_entry = function
    | Class name_index ->
        (match get_entry name_index with
        | UTF8 _ -> ()
        | _ -> fail (Malformed_Class_entry name_index))
    | Fieldref (class_index, name_and_type_index) ->
        (match (get_entry class_index), (get_entry name_and_type_index) with
        | (Class _), (NameAndType _) -> ()
        | _ -> fail (Malformed_Fieldref_entry (class_index, name_and_type_index)))
    | Methodref (class_index, name_and_type_index) ->
        (match (get_entry class_index), (get_entry name_and_type_index) with
        | (Class _), (NameAndType _) -> ()
        | _ -> fail (Malformed_Methodref_entry (class_index, name_and_type_index)))
    | InterfaceMethodref (class_index, name_and_type_index) ->
        (match (get_entry class_index), (get_entry name_and_type_index) with
        | (Class _), (NameAndType _) -> ()
        | _ -> fail (Malformed_InterfaceMethodRef_entry (class_index, name_and_type_index)))
    | String value_index ->
        (match get_entry value_index with
        | UTF8 _ -> ()
        | _ -> fail (Malformed_String_entry value_index))
    | Integer _ -> ()
    | Float _ -> ()
    | Long _ -> ()
    | Double _ -> ()
    | NameAndType (name_index, desc_index) ->
        (match (get_entry name_index), (get_entry desc_index) with
        | (UTF8 _), (UTF8 _) -> ()
        | _ -> fail (Malformed_NameAndType_entry (name_index, desc_index)))
    | UTF8 _ -> ()
    | MethodHandle (reference_kind, reference_index) ->
        (match reference_kind, (get_entry reference_index) with
        | Reference.REF_getField, (Fieldref _)
        | Reference.REF_getStatic, (Fieldref _)
        | Reference.REF_putField, (Fieldref _)
        | Reference.REF_putStatic, (Fieldref _)
        | Reference.REF_invokeVirtual, (Methodref _)
        | Reference.REF_invokeStatic, (Methodref _ | InterfaceMethodref _)
        | Reference.REF_invokeSpecial, (Methodref _ | InterfaceMethodref _)
        | Reference.REF_newInvokeSpecial, (Methodref _)
        | Reference.REF_invokeInterface, (InterfaceMethodref _) -> ()
        | _ -> fail (Malformed_MethodHandle_entry (reference_kind, reference_index)))
    | MethodType type_index ->
        (match get_entry type_index with
        | UTF8 _ -> ()
        | _ -> fail (Malformed_MethodType_entry type_index))
    | InvokeDynamic (index, name_and_type_index) ->
        (match get_entry name_and_type_index with
        | NameAndType _ -> ()
        | _ -> fail (Malformed_InvokeDynamic_entry (index, name_and_type_index)))
    | ModuleId (name_index, vers_index) ->
        (match (get_entry name_index), (get_entry vers_index) with
        | (UTF8 _), (UTF8 _) -> ()
        | _ -> fail (Malformed_ModuleId_entry (name_index, vers_index))) in
  Array.iter check_entry pool

let check_entry_for_kind cpool idx tag =
  try
    match tag, (get_entry cpool idx) with
    | CONSTANT_Class, Class _
    | CONSTANT_Fieldref, Fieldref _
    | CONSTANT_Methodref, Methodref _
    | CONSTANT_InterfaceMethodref, InterfaceMethodref _
    | CONSTANT_String, String _
    | CONSTANT_Integer, Integer _
    | CONSTANT_Float, Float _
    | CONSTANT_Long, Long _
    | CONSTANT_Double, Double _
    | CONSTANT_NameAndType, NameAndType _
    | CONSTANT_Utf8, UTF8 _
    | CONSTANT_MethodHandle, MethodHandle _
    | CONSTANT_MethodType, MethodType _
    | CONSTANT_InvokeDynamic, InvokeDynamic _
    | CONSTANT_ModuleId, ModuleId _ -> true
    | _ -> false
  with _ -> false

let version_bounds = function
  | Class _ ->
      Version.make_bounds "'Class' constant pool element" Version.Java_1_0 None
  | Fieldref _ ->
      Version.make_bounds "'Fieldref' constant pool element" Version.Java_1_0 None
  | Methodref _ ->
      Version.make_bounds "'Methodref' constant pool element" Version.Java_1_0 None
  | InterfaceMethodref _ ->
      Version.make_bounds "'InterfaceMethodref' constant pool element" Version.Java_1_0 None
  | String _ ->
      Version.make_bounds "'String' constant pool element" Version.Java_1_0 None
  | Integer _ ->
      Version.make_bounds "'Integer' constant pool element" Version.Java_1_0 None
  | Float _ ->
      Version.make_bounds "'Float' constant pool element" Version.Java_1_0 None
  | Long _ ->
      Version.make_bounds "'Long' constant pool element" Version.Java_1_0 None
  | Double _ ->
      Version.make_bounds "'Double' constant pool element" Version.Java_1_0 None
  | NameAndType _ ->
      Version.make_bounds "'NameAndType' constant pool element" Version.Java_1_0 None
  | UTF8 _ ->
      Version.make_bounds "'UTF8' constant pool element" Version.Java_1_0 None
  | MethodHandle _ ->
      Version.make_bounds "'MethodHandle' constant pool element" Version.Java_1_7 None
  | MethodType _ ->
      Version.make_bounds "'MethodType' constant pool element" Version.Java_1_7 None
  | InvokeDynamic _ ->
      Version.make_bounds "'InvokeDynamic' constant pool element" Version.Java_1_7 None
  | ModuleId _ ->
      Version.make_bounds "'ModuleId' constant pool element" Version.Java_1_9 None

let check_version v pool =
  for i = 1 to pred (Array.length pool) do
    Version.check (version_bounds pool.(i)) v
  done


(* Extendable pools *)

module ExtendableArray =
  InvertibleArray.Make
    (struct
      type t = element
      let equal = equal_element
      let compare = compare_element
      let hash = hash_element
      let to_string = to_string_element
    end)

type extendable = ExtendableArray.t

let make_extendable () =
  ExtendableArray.make 1 128 dummy_element

let make_extendable_from_pool pool =
  let error = Too_large (Array.length pool) in
  ExtendableArray.from_array (Exception error) pool dummy_element

let make_extendable_from_array array =
  make_extendable_from_pool array

let get_extendable_entry pool i =
  let i' = (i : Utils.u2 :> int) in
  let len = ExtendableArray.length pool in
  if i' = 0 then
    fail Invalid_reference
  else if (i' < 0) || (i' >= len) then
    fail (Reference_out_of_bounds (i', len))
  else
    let res = ExtendableArray.get pool i in
    if res == dummy_element then
      fail (Dummy_access i)
    else
      res

let add_if_not_found ext elem =
  ExtendableArray.add_if_not_found
    (Exception (Too_large (ExtendableArray.length ext)))
    ext
    elem
    dummy_element
    (element_size elem = 2)

let add_utf8 ext s =
  let elem = UTF8 s in
  if UTF8.equal s @"Dummy-Constant-Pool-Entry" then
    (* in this case we actually want to emit UTF "Dummy-Constant-Pool-Entry" *)
    ExtendableArray.add
      (Exception (Too_large (ExtendableArray.length ext)))
      ext
      elem
      dummy_element
      (element_size elem = 2)
  else
    add_if_not_found ext elem

let add_class ext n =
  let name_index = add_utf8 ext (Name.internal_utf8_for_class n) in
  let elem = Class name_index in
  add_if_not_found ext elem

let add_array_class ext d =
  let name_index = add_utf8 ext (Descriptor.internal_utf8_of_java_type (d :> Descriptor.java_type)) in
  let elem = Class name_index in
  add_if_not_found ext elem

let add_name_and_type ext n t =
  let name_index = add_utf8 ext n in
  let type_index = add_utf8 ext t in
  let elem = NameAndType (name_index, type_index) in
  add_if_not_found ext elem

let add_moduleid ext n v =
  let name_index = add_utf8 ext n in
  let version_index = add_utf8 ext v in
  let elem = ModuleId (name_index, version_index) in
  add_if_not_found ext elem

let add_field ext cn n t =
  let class_index = add_class ext cn in
  let d = Descriptor.utf8_of_field t in
  let name_and_type_index = add_name_and_type ext (Name.utf8_for_field n) d in
  let elem = Fieldref (class_index, name_and_type_index) in
  add_if_not_found ext elem

let add_method ext cn n t =
  let class_index = add_class ext cn in
  let d = Descriptor.utf8_of_method t in
  let name_and_type_index = add_name_and_type ext (Name.utf8_for_method n) d in
  let elem = Methodref (class_index, name_and_type_index) in
  add_if_not_found ext elem

let add_interface_method ext cn n t =
  let class_index = add_class ext cn in
  let d = Descriptor.utf8_of_method t in
  let name_and_type_index = add_name_and_type ext (Name.utf8_for_method n) d in
  let elem = InterfaceMethodref (class_index, name_and_type_index) in
  add_if_not_found ext elem

let add_array_method ext at n t =
  let class_index = add_array_class ext at in
  let d = Descriptor.utf8_of_method t in
  let name_and_type_index = add_name_and_type ext (Name.utf8_for_method n) d in
  let elem = Methodref (class_index, name_and_type_index) in
  add_if_not_found ext elem

let add_string ext s =
  let v = add_utf8 ext s in
  let elem = String v in
  add_if_not_found ext elem

let add_integer ext i =
  let elem = Integer i in
  add_if_not_found ext elem

let add_float ext f =
  let elem = Float (Int32.bits_of_float f) in
  add_if_not_found ext elem

let add_long ext l =
  let high, low = Utils.(split_s8 (s8 l)) in
  let elem = Long ((high :> int32), (low :> int32)) in
  add_if_not_found ext elem

let add_double ext d =
  let l = Int64.bits_of_float d in
  let high, low = Utils.(split_s8 (s8 l)) in
  let elem = Double ((high :> int32), (low :> int32)) in
  add_if_not_found ext elem

let add_method_handle ext r =
  let elem = match r with
  | Reference.GetField (cn, fn, fd) ->
      let reference_index = add_field ext cn fn fd in
      MethodHandle (Reference.REF_getField, reference_index)
  | Reference.GetStatic (cn, fn, fd) ->
      let reference_index = add_field ext cn fn fd in
      MethodHandle (Reference.REF_getStatic, reference_index)
  | Reference.PutField (cn, fn, fd) ->
      let reference_index = add_field ext cn fn fd in
      MethodHandle (Reference.REF_putField, reference_index)
  | Reference.PutStatic (cn, fn, fd) ->
      let reference_index = add_field ext cn fn fd in
      MethodHandle (Reference.REF_putStatic, reference_index)
  | Reference.InvokeVirtual (cn, mn, mt) ->
      let reference_index = add_method ext cn mn mt in
      MethodHandle (Reference.REF_invokeVirtual, reference_index)
  | Reference.InvokeStatic (cn, mn, mt) ->
      let reference_index = add_method ext cn mn mt in
      MethodHandle (Reference.REF_invokeStatic, reference_index)
  | Reference.InvokeSpecial (cn, mn, mt) ->
      let reference_index = add_method ext cn mn mt in
      MethodHandle (Reference.REF_invokeSpecial, reference_index)
  | Reference.NewInvokeSpecial (cn, pl) ->
      let mn = Name.make_for_method class_constructor in
      let mt = pl, (`Class cn) in
      let reference_index = add_method ext cn mn mt in
      MethodHandle (Reference.REF_newInvokeSpecial, reference_index)
  | Reference.InvokeInterface (cn, mn, mt) ->
      let reference_index = add_interface_method ext cn mn mt in
      MethodHandle (Reference.REF_invokeInterface, reference_index) in
  add_if_not_found ext elem

let add_method_type ext t =
  let d = Descriptor.utf8_of_method t in
  let type_index = add_utf8 ext d in
  let elem = MethodType type_index in
  add_if_not_found ext elem

let add_invoke_dynamic ext index mn md =
  let n = Name.utf8_for_method mn in
  let t = Descriptor.utf8_of_method md in
  let name_and_type_index = add_name_and_type ext n t in
  let elem = InvokeDynamic (index, name_and_type_index) in
  add_if_not_found ext elem

let to_pool ext =
  ExtendableArray.to_array ext
