(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Methods in both low- and high-level forms.

    It also provides conversion functions between levels as well as i/o
    functions for low-level. *)


(** {6 Low-level form} *)

type info = {
    access_flags : Utils.u2;
    name_index : Utils.u2;
    descriptor_index : Utils.u2;
    attributes_count : Utils.u2;
    attributes_array : Attribute.info array;
  }
(** Represents a method as defined in the class file format specification. *)


(** {6 Exception} *)

BARISTA_ERROR =
  | Invalid_name of UTF8.t
  | Invalid_name_value of Utils.u2
  | Invalid_descriptor_value of Utils.u2


(** {6 I/O functions} *)

val read_info : InputStream.t -> info
(** [read_info st] reads a method from [st].
    Raises [InputStream.Exception] if an i/o error occurs. *)

val write_info : OutputStream.t -> info -> unit
(** [write_info st m] writes method [m] onto [st].
    Raises [OutputStream.Exception] if an i/o error occurs. *)


(** {6 High-level form} *)

type regular = {
    flags : AccessFlag.for_method list;
    name : Name.for_method;
    descriptor : Descriptor.for_method;
    attributes : Attribute.for_method list;
  }
(** Represents a {i regular} (possibly static) method. *)

type constructor = {
    cstr_flags : AccessFlag.for_constructor list;
    cstr_descriptor : Descriptor.for_parameter list;
    cstr_attributes : Attribute.for_method list;
  }
(** Represents an instance constructor method. *)

type class_initializer = {
    init_flags : AccessFlag.for_initializer list;
    init_attributes : Attribute.for_method list;
  }
(** Represents a class initializer method. *)

type t =
  | Regular of regular (** Regular method. *)
  | Constructor of constructor (** Instance constructor. *)
  | Initializer of class_initializer (** Class initializer. *)
(** Represents the different kinds of methods. *)

val equal_regular : regular -> regular -> bool
(** Equality over regular methods. *)

val compare_regular : regular -> regular -> int
(** Comparison over regular methods. *)

val hash_regular : regular -> int
(** Hash function over regular methods. *)

val equal_constructor : constructor -> constructor -> bool
(** Equality over constructors. *)

val compare_constructor : constructor -> constructor -> int
(** Comparison over constructors. *)

val hash_constructor : constructor -> int
(** Hash function over constructors. *)

val equal_class_initializer : class_initializer -> class_initializer -> bool
(** Equality over class initializers. *)

val compare_class_initializer : class_initializer -> class_initializer -> int
(** Comparison over class initializers. *)

val hash_class_initializer : class_initializer -> int
(** Hash function over class initializers. *)

val equal : t -> t -> bool
(** Equality over methods. *)

val compare : t -> t -> int
(** Comparison over methods. *)

val hash : t -> int
(** Hash function over methods. *)

val compare_according_to_visibility : t -> t -> int
(** Comparison over methods, to be used to compare for user-intended
    output. *)


(** {6 Conversion functions} *)

val decode : Version.t -> bool -> Bootstrap.methods -> ConstantPool.t -> info -> t
(** Converts from a low-level into a high-level form according to passed
    pool, and bootstrap method information. The first parameter indicates
    whether the enclosing element is an interface or a class.
    Raises [Exception] if an error occurs during conversion. *)

val encode : Bootstrap.methods -> ConstantPool.extendable -> t -> info
(** Converts from a high-level into a low-level form, using passed
    extendable pool, and bootstrap method information.
    Raises [Exception] if an error occurs during conversion. *)
