(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


let (++) = UTF8.(++)

type t = {
    class_path : ClassPath.t;
    loaded_classes : ClassDefinition.t UTF8.Hashtbl.t;
    loaded_packages : PackageDefinition.t UTF8.Hashtbl.t;
    loaded_modules : ModuleDefinition.t UTF8.Hashtbl.t;
  }

BARISTA_ERROR =
  | Unable_to_load of (s : UTF8.t) * (err : string) ->
      Printf.sprintf "unable to load class %S (%s)"
        (UTF8.to_string_noerr s)
        err
  | Already_defined of (s : UTF8.t) ->
      Printf.sprintf "class %S is already defined"
        (UTF8.to_string_noerr s)

let make_of_class_path cp =
  { class_path = cp;
    loaded_classes = UTF8.Hashtbl.create 997;
    loaded_packages = UTF8.Hashtbl.create 79;
    loaded_modules = UTF8.Hashtbl.create 79; }

let create_functions
    ~get_table (* (t -> 'a UTF8.Hashtbl.t) *)
    ~name_suffix (* UTF8.t *)
    ~decode_func (* ClassFile.t -> 'a *)
    ~name_of_element (* 'a -> 'b *)
    ~utf8_of_name (* 'b -> UTF8.t *) =
  let find loader name =
    let table = get_table loader in
    try
      UTF8.Hashtbl.find table name
    with Not_found ->
      let stream =
        ClassPath.open_stream loader.class_path (name ++ name_suffix) in
      try
        let def =
          stream
          |> ClassFile.read
          |> decode_func in
        InputStream.close_noerr stream;
        UTF8.Hashtbl.add table name def;
        def
      with
      | ClassFile.Exception cause ->
          InputStream.close_noerr stream;
          fail (Unable_to_load (name, ClassFile.string_of_error cause))
      | e ->
          InputStream.close_noerr stream;
          fail (Unable_to_load (name, Printexc.to_string e)) in
  let find_name loader name =
    find loader (utf8_of_name name) in
  let add loader def =
    let table = get_table loader in
    let name = utf8_of_name (name_of_element def) in
    if UTF8.Hashtbl.mem table name then
      fail (Already_defined name)
    else
      UTF8.Hashtbl.add table name def in
  let mem loader name =
    UTF8.Hashtbl.mem (get_table loader) name in
  let mem_name loader name =
    mem loader (utf8_of_name name) in
  let remove loader name =
    UTF8.Hashtbl.remove (get_table loader) name in
  let remove_name loader name =
    remove loader (utf8_of_name name) in
  let replace loader def =
    let name =
      def
      |> name_of_element
      |> utf8_of_name in
    UTF8.Hashtbl.replace (get_table loader) name def in
  find, find_name, add, mem, mem_name, remove, remove_name, replace

let find_class,
  find_class_name,
  add_class,
  mem_class,
  mem_class_name,
  remove_class,
  remove_class_name,
  replace_class =
  create_functions
    ~get_table:(fun cl -> cl.loaded_classes)
    ~name_suffix:@""
    ~decode_func:ClassDefinition.decode
    ~name_of_element:(fun cd -> cd.ClassDefinition.name)
    ~utf8_of_name:Name.external_utf8_for_class

let find_package,
  find_package_name,
  add_package,
  mem_package,
  mem_package_name,
  remove_package,
  remove_package_name,
  replace_package =
  create_functions
    ~get_table:(fun cl -> cl.loaded_packages)
    ~name_suffix:@"/package-info"
    ~decode_func:PackageDefinition.decode
    ~name_of_element:(fun cd -> cd.PackageDefinition.name)
    ~utf8_of_name:Name.external_utf8_for_package

let find_module,
  find_module_name,
  add_module,
  mem_module,
  mem_module_name,
  remove_module,
  remove_module_name,
  replace_module =
  create_functions
    ~get_table:(fun cl -> cl.loaded_modules)
    ~name_suffix:@"/module-info"
    ~decode_func:ModuleDefinition.decode
    ~name_of_element:(fun cd -> cd.ModuleDefinition.name)
    ~utf8_of_name:Name.external_utf8_for_module
