(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

IFNDEF USE_JDK THEN

type t = {
    zip : Zip.in_file;
    mutable iterated : bool;
  }

BARISTA_ERROR =
  | Unable_to_open_archive of (p : Path.t) ->
      Printf.sprintf "unable to open archive (%S)"
        (Path.to_string p)      
  | Unable_to_read_entry of (n : UTF8.t) ->
      Printf.sprintf "unable to read entry (%S)" (UTF8.to_string_noerr n)
  | Unable_to_close_archive ->
      "unable to close archive"

external bytes_of_string : string -> Bytes.t =
  "%identity"

external archive_entry_of_zip_entry : Zip.entry -> ArchiveEntry.t =
  "%identity"

let make_of_path path =
  try
    { zip = path
            |> Path.to_string
            |> Zip.open_in;
      iterated = false; }
  with _ ->
    fail (Unable_to_open_archive path)

let iter_entries f archive =
  if not archive.iterated then begin
    archive.iterated <- true;
    archive.zip
    |> (try Zip.entries with _ -> fail (Unable_to_read_entry @"<entry-list>"))
    |> List.iter
        (fun entry ->
          let bytes =
            try
              entry
              |> Zip.read_entry archive.zip
              |> bytes_of_string
            with _ ->
              fail (Unable_to_read_entry (UTF8.of_string entry.Zip.filename)) in
          f (archive_entry_of_zip_entry entry) bytes)
  end

let fold_entries f zero archive =
  if not archive.iterated then begin
    archive.iterated <- true;
    archive.zip
    |> (try Zip.entries with _ -> fail (Unable_to_read_entry @"<entry-list>"))
    |> List.fold_left
        (fun acc entry ->
          let bytes =
            try
              entry
              |> Zip.read_entry archive.zip
              |> bytes_of_string
            with _ ->
              fail (Unable_to_read_entry (UTF8.of_string entry.Zip.filename)) in
          f acc (archive_entry_of_zip_entry entry) bytes)
        zero
  end else
    zero

let close archive =
  try
    Zip.close_in archive.zip
  with _ ->
    fail Unable_to_close_archive

let close_noerr archive =
  try
    Zip.close_in archive.zip
  with _ ->
    ()

ELSE (* USE_JDK *)

type t = java'util'zip'ZipInputStream java_instance

BARISTA_ERROR =
  | Unable_to_open_archive of (p : Path.t) ->
      Printf.sprintf "unable to open archive (%S)"
        (Path.to_string p)      
  | Unable_to_read_entry of (n : UTF8.t) ->
      Printf.sprintf "unable to read entry (%S)" (UTF8.to_string_noerr n)
  | Unable_to_close_archive ->
      "unable to close archive"

external file_of_path : Path.t -> java'io'File java_instance =
  "%identity"

external archive_entry_of_zip_entry : java'util'zip'ZipEntry java_instance -> ArchiveEntry.t =
  "%identity"

external bytes_of_byte_array : int JavaByteArray.t -> Bytes.t =
  "%identity"

external utf8_of_java_string : java'lang'String java_instance -> UTF8.t =
  "%identity"

let make_of_path path =
  let file =
    path
    |> file_of_path in
  let files_state =
    Java.call "org.ocamljava.runtime.context.CurrentContext.getFilesState()" () in
  let hook =
    Java.call "org.ocamljava.runtime.context.FilesState.getFileHook()" files_state in
  let stream =
    if Java.is_not_null hook then
      Some (file
           |> Java.call "org.ocamljava.runtime.context.FilesState.resourceNameFromFile(_)" files_state
           |> Java.call "org.ocamljava.runtime.context.FileHook.getInputStream(_)" hook)
    else
      None in
  match stream with
  | Some st when Java.is_not_null st ->
      st
      |> Java.make "java.util.zip.ZipInputStream(_)"
  | Some _ | None ->
      try
        file
        |> Java.make "java.io.FileInputStream(java.io.File)"
        |> Java.make "java.util.zip.ZipInputStream(_)"
      with Java_exception _ ->
        fail (Unable_to_open_archive path)

let read_bytes archive filename =
  let buffer_size = 65536l in
  try
    let buffer = Java.make_array "byte[]" 65536l in
    let data = Java.make "java.io.ByteArrayOutputStream(_)" buffer_size in
    let last_read = ref buffer_size in
    while !last_read <> -1l do
      last_read :=
        Java.call "java.util.zip.ZipInputStream.read(_,_,_)" archive buffer 0l buffer_size;
      if !last_read <> -1l then
        Java.call "java.io.ByteArrayOutputStream.write(_,_,_)" data buffer 0l !last_read;
    done;
    data
    |> Java.call "java.io.ByteArrayOutputStream.toByteArray()"
    |> bytes_of_byte_array
  with _ ->
    fail (Unable_to_read_entry filename)

let iter_entries f archive =
  let entry = ref (Java.call "java.util.zip.ZipInputStream.getNextEntry()" archive) in
  while Java.is_not_null !entry do
    let filename =
      !entry
      |> Java.call "java.util.zip.ZipEntry.getName()"
      |> utf8_of_java_string in
    f (archive_entry_of_zip_entry !entry) (read_bytes archive filename);
    Java.call "java.util.zip.ZipInputStream.closeEntry()" archive;
    entry := Java.call "java.util.zip.ZipInputStream.getNextEntry()" archive
  done

let fold_entries f zero archive =
  let acc = ref zero in
  let entry = ref (Java.call "java.util.zip.ZipInputStream.getNextEntry()" archive) in
  while Java.is_not_null !entry do
    let filename =
      !entry
      |> Java.call "java.util.zip.ZipEntry.getName()"
      |> utf8_of_java_string in
    acc := f !acc (archive_entry_of_zip_entry !entry) (read_bytes archive filename);
    Java.call "java.util.zip.ZipInputStream.closeEntry()" archive;
    entry := Java.call "java.util.zip.ZipInputStream.getNextEntry()" archive
  done;
  !acc

let close archive =
  try
    archive
    |> Java.call "java.util.zip.ZipInputStream.close()"
  with Java_exception _ ->
    fail Unable_to_close_archive

let close_noerr archive =
  try
    archive
    |> Java.call "java.util.zip.ZipInputStream.close()"
  with Java_exception _ ->
    ()

END
