(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Output streams.

    As stated in the class file format definition, {i multibyte data
    items are always stored in big-endian order, where the high bytes
    come first}.

    This definition is independent of the actual implementation, that can
    be based on either the OCaml standard library, or the OCaml-Java
    runtime library. *)


(** {6 Type} *)

type t
(** The type of output streams, whatever the stream destination is. *)


(** {6 Exception} *)

BARISTA_ERROR =
  | Unable_to_open_stream of Path.t
  | Unable_to_write_data
  | Unable_to_close_stream


(** {6 Constructors} *)

val make_of_buffer : ByteBuffer.t -> t
(** Creates an output stream whose destination is a buffer. *)

val make_of_channel : out_channel -> t
(** Creates an output stream whose destination is a channel. *)

val make_of_path : Path.t -> t
(** Creates an output stream whose destination is a path.

    Raises [Exception] stream cannot be opened. *)


(** {6 Functions} *)

val write_u1 : t -> Utils.u1 -> unit
(** Writes an unsigned 1-byte integer to the passed stream.

    Raises [Exception] if data cannot be written. *)

val write_u2 : t -> Utils.u2 -> unit
(** Writes an unsigned 2-byte integer to the passed stream.

    Raises [Exception] if data cannot be written. *)

val write_u4 : t -> Utils.u4 -> unit
(** Writes an unsigned 4-byte integer to the passed stream.

    Raises [Exception] if data cannot be written. *)

val write_s1 : t -> Utils.s1 -> unit
(** Writes a signed 1-byte integer to the passed stream.

    Raises [Exception] if data cannot be written. *)

val write_s2 : t -> Utils.s2 -> unit
(** Writes a signed 2-byte integer to the passed stream.

    Raises [Exception] if data cannot be written. *)

val write_s4 : t -> Utils.s4 -> unit
(** Writes a signed 4-byte integer to the passed stream.

    Raises [Exception] if data cannot be written. *)

val write_s8 : t -> Utils.s8 -> unit
(** Writes a signed 8-byte integer to the passed stream.

    Raises [Exception] if data cannot be written. *)

val write_bytes_from : t -> Bytes.t -> int -> int -> unit
(** [write_bytes_from stream bytes pos len] writes [len] bytes from
    [bytes] starting at [pos] to [stream].

    Raises [Exception] if data cannot be written. *)

val write_bytes : t -> Bytes.t -> unit
(** [write_bytes stream bytes] writes [bytes] to [stream].

    Raises [Exception] if data cannot be written. *)

val write_utf8 : t -> UTF8.t -> unit
(** [write_utf8 st str] first converts [str] into bytes, using
    {i modified} format, as they appear in Java class files (cf. the
    documentation of the {i java.io.DataInput} class). Then, writes the
    size of the resulting bytes to the passed stream as a 2-byte integer.
    Finally, writes the bytes to the stream. *)

val write_elements : exn -> t -> (t -> 'a -> unit) -> 'a list -> unit
(** [write_elements e stream f l] will first write to [stream] the length
    of [l] as a [Utils.u2] value. Then, it will write the elements from
    [l] using [f].

    Raises [e] if the length of [l] cannot be encoded as a [Utils.u2]
    value.

    Raises [Exception] if data cannot be written. *)

val flush : t -> unit
(** Flushes the passed stream.

    Raises [Exception] if an error occurs while flushing the stream. *)

val close : t -> unit
(** Closes the passed stream, any subsequent write will fail raising an
    exception.

    Raises [Exception] if an error occurs while closing the stream. *)

val close_noerr : t -> unit
(** Same as [close] but raised exceptions are silently discarded. *)

val try_with : t -> (t -> 'a) -> 'a
(** [try_with stream f] is equivalent to
    [Utils.try_finally stream f close_noerr]. *)


(** {6 Predefined streams} *)

val stdout : t
(** Predefined stream for standard output. *)

val stderr : t
(** Predefined stream for standard error. *)
