(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(* IMPLEMENTATION NOTE: some code is manually "inlined" in order to avoid
   method calls, and hence multiple method dispatchs for a single
   "top-level" call (hopefully resulting in better performance). *)

IFNDEF USE_JDK THEN

BARISTA_ERROR =
  | End_of_lexer -> "end of lexer"
  | Invalid_consume of (expected : UChar.t) * (found : UChar.t) ->
      Printf.sprintf "invalid consume (%C waited but %C found)"
        (UChar.to_char_noerr expected)
        (UChar.to_char_noerr found)

external uchar_of_camomile : CamomileLibrary.UChar.t -> UChar.t =
  "%identity"

external camomile_of_utf8 : UTF8.t -> CamomileLibrary.UTF8.t =
  "%identity"

external utf8_of_camomile : CamomileLibrary.UTF8.t -> UTF8.t =
  "%identity"

let default_buffer_size = 256

class t str =
  let str = camomile_of_utf8 str in
  object (self)

    val mutable next = CamomileLibrary.UTF8.first str

    method is_available =
      not (CamomileLibrary.UTF8.out_of_range str next)

    method peek =
      if CamomileLibrary.UTF8.out_of_range str next then
        fail End_of_lexer
      else
        CamomileLibrary.UTF8.look str next
        |> uchar_of_camomile

    method look_ahead_string chars =
      if CamomileLibrary.UTF8.out_of_range str next then
        fail End_of_lexer
      else begin
        let next_char =
          CamomileLibrary.UTF8.look str next
          |> uchar_of_camomile in
        UTF8.contains next_char chars
      end

    method look_ahead_list l =
      if CamomileLibrary.UTF8.out_of_range str next then
        fail End_of_lexer
      else begin
        let next_char =
          CamomileLibrary.UTF8.look str next
          |> uchar_of_camomile in
        List.exists (fun ch -> UChar.equal ch next_char) l
      end

    method look_ahead ch =
      if CamomileLibrary.UTF8.out_of_range str next then
        fail End_of_lexer
      else begin
        let next_char =
          CamomileLibrary.UTF8.look str next
          |> uchar_of_camomile in
        UChar.equal next_char ch
      end

    method consume_char =
      if CamomileLibrary.UTF8.out_of_range str next then
        fail End_of_lexer
      else begin
        let res = CamomileLibrary.UTF8.look str next in
        next <- CamomileLibrary.UTF8.next str next;
        uchar_of_camomile res
      end

    method consume =
      if CamomileLibrary.UTF8.out_of_range str next then
        fail End_of_lexer
      else
        next <- CamomileLibrary.UTF8.next str next

    method consume_only ch =
      if CamomileLibrary.UTF8.out_of_range str next then
        fail End_of_lexer
      else begin
        let next_char =
          CamomileLibrary.UTF8.look str next
          |> uchar_of_camomile in
        next <- CamomileLibrary.UTF8.next str next;
        if not (UChar.equal ch next_char) then
          fail (Invalid_consume (ch, next_char))
      end

    method consume_until_string chars =
      let buff = CamomileLibrary.UTF8.Buf.create default_buffer_size in
      while not (self#look_ahead_string chars) do
        CamomileLibrary.UTF8.look str next
        |> CamomileLibrary.UTF8.Buf.add_char buff;
        next <- CamomileLibrary.UTF8.next str next
      done;
      CamomileLibrary.UTF8.Buf.contents buff
      |> utf8_of_camomile

    method consume_until_list l =
      let buff = CamomileLibrary.UTF8.Buf.create default_buffer_size in
      while not (self#look_ahead_list l) do
        CamomileLibrary.UTF8.look str next
        |> CamomileLibrary.UTF8.Buf.add_char buff;
        next <- CamomileLibrary.UTF8.next str next
      done;
      CamomileLibrary.UTF8.Buf.contents buff
      |> utf8_of_camomile

    method consume_until ch =
      let buff = CamomileLibrary.UTF8.Buf.create default_buffer_size in
      while not (self#look_ahead ch) do
        CamomileLibrary.UTF8.look str next
        |> CamomileLibrary.UTF8.Buf.add_char buff;
        next <- CamomileLibrary.UTF8.next str next
      done;
      CamomileLibrary.UTF8.Buf.contents buff
      |> utf8_of_camomile

    method consume_all =
      let buff = CamomileLibrary.UTF8.Buf.create default_buffer_size in
      while not (CamomileLibrary.UTF8.out_of_range str next) do
        CamomileLibrary.UTF8.look str next
        |> CamomileLibrary.UTF8.Buf.add_char buff;
        next <- CamomileLibrary.UTF8.next str next
      done;
      CamomileLibrary.UTF8.Buf.contents buff
      |> utf8_of_camomile

    method consume_whitespace =
      while (not (CamomileLibrary.UTF8.out_of_range str next))
          && (UChar.is_whitespace
                (uchar_of_camomile (CamomileLibrary.UTF8.look str next))) do
        next <- CamomileLibrary.UTF8.next str next
      done

  end

ELSE (* USE_JDK *)

BARISTA_ERROR =
  | End_of_lexer -> "end of lexer"
  | Invalid_consume of (expected : UChar.t) * (found : UChar.t) ->
      Printf.sprintf "invalid consume (%C waited but %C found)"
        (UChar.to_char_noerr expected)
        (UChar.to_char_noerr found)

external uchar_of_code_point : int32 -> UChar.t =
  "%identity"

external code_point_of_uchar : UChar.t -> int32 =
  "%identity"

external java_string_of_utf8 : UTF8.t -> java'lang'String java_instance =
  "%identity"

external utf8_of_java_string : java'lang'String java_instance -> UTF8.t =
  "%identity"

let default_buffer_size = 256l

class t str =
  let str = java_string_of_utf8 str in
  let len = Java.call "String.length()" str in
  object (self)

    val mutable next = 0l

    method is_available =
      next < len

    method peek =
      if next >= len then
        fail End_of_lexer
      else
        Java.call "String.codePointAt(_)" str next
        |> uchar_of_code_point

    method look_ahead_string chars =
      if next >= len then
        fail End_of_lexer
      else begin
        let chars = java_string_of_utf8 chars in
        let next_char = Java.call "String.codePointAt(_)" str next in
        (Java.call "String.indexOf(int)" chars next_char) >= 0l
      end

    method look_ahead_list l =
      if next >= len then
        fail End_of_lexer
      else begin
        let next_char = Java.call "String.codePointAt(_)" str next in
        List.exists (fun ch -> (code_point_of_uchar ch) = next_char) l
      end

    method look_ahead ch =
      if next >= len then
        fail End_of_lexer
      else begin
        let next_char = Java.call "String.codePointAt(_)" str next in
        next_char = (code_point_of_uchar ch)
      end

    method consume_char =
      if next >= len then
        fail End_of_lexer
      else begin
        let res = Java.call "String.codePointAt(_)" str next in
        next <- Int32.succ next;
        uchar_of_code_point res
      end

    method consume =
      if next >= len then
        fail End_of_lexer
      else
        next <- Int32.succ next;

    method consume_only ch =
      if next >= len then
        fail End_of_lexer
      else begin
        let next_char = Java.call "String.codePointAt(_)" str next in
        next <- Int32.succ next;
        if next_char <> (code_point_of_uchar ch) then
          fail (Invalid_consume (ch, uchar_of_code_point next_char))
      end

    method consume_until_string chars =
      let buff = Java.make "StringBuilder(int)" default_buffer_size in
      while not (self#look_ahead_string chars) do
        Java.call "String.codePointAt(_)" str next
        |> Java.call "StringBuilder.appendCodePoint(_):StringBuilder" buff
        |> ignore;
        next <- Int32.succ next
      done;
      Java.call "StringBuilder.toString()" buff
      |> utf8_of_java_string

    method consume_until_list l =
      let buff = Java.make "StringBuilder(int)" default_buffer_size in
      while not (self#look_ahead_list l) do
        Java.call "String.codePointAt(_)" str next
        |> Java.call "StringBuilder.appendCodePoint(_):StringBuilder" buff
        |> ignore;
        next <- Int32.succ next
      done;
      Java.call "StringBuilder.toString()" buff
      |> utf8_of_java_string

    method consume_until ch =
      let buff = Java.make "StringBuilder(int)" default_buffer_size in
      while not (self#look_ahead ch) do
        Java.call "String.codePointAt(_)" str next
        |> Java.call "StringBuilder.appendCodePoint(_):StringBuilder" buff
        |> ignore;
        next <- Int32.succ next
      done;
      Java.call "StringBuilder.toString()" buff
      |> utf8_of_java_string

    method consume_all =
      let buff = Java.make "StringBuilder(int)" default_buffer_size in
      while next < len do
        Java.call "String.codePointAt(_)" str next
        |> Java.call "StringBuilder.appendCodePoint(_):StringBuilder" buff
        |> ignore;
        next <- Int32.succ next
      done;
      Java.call "StringBuilder.toString()" buff
      |> utf8_of_java_string

    method consume_whitespace =
      while (next < len)
          && (Java.call "Character.isWhitespace(int)"
                (Java.call "String.codePointAt(_)" str next)) do
        next <- Int32.succ next
      done

  end

END
