(*
 * This file is part of OCaml-Java wrapper.
 * Copyright (C) 2007-2015 Xavier Clerc.
 *
 * OCaml-Java wrapper is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * OCaml-Java wrapper is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Representation of Java code fragment.

    Does not support the whole language, but only the elements needed
    by code generated by ocamlwrap. *)

val checked_name : string -> string

type modifier =
  | Public
  | Protected
  | Private
  | Static
  | Abstract
  | Final
  | Native
  | Synchronized
  | Transient
  | Volatile
  | Strictfp

type type_ =
  | Byte
  | Short
  | Char
  | Int
  | Long
  | Float
  | Double
  | Boolean
  | Reference of string * (type_ list)
  | Array of type_

val string_of_type : type_ -> string

type expression =
  | Call of expression * string * (expression list)
  | Static_call of string * string * (expression list)
  | Identifier of string
  | String_literal of string
  | Int_literal of int32
  | Boolean_literal of bool
  | Null_literal
  | Anonymous_class of type_ * (expression list) * type_ * string * ((type_ * string) list) * expression
  | New of string * expression list
  | New_array of type_ * expression list
  | Cast of type_ * expression
  | Instance_of of expression * type_
  | Infix of string * expression * expression

type statement =
  | Expression of expression
  | Variable_declaration of type_ * string * expression
  | Return of expression option
  | Super_constructor of expression list
  | Assign of string * expression
  | If of expression * statement
  | If_else of expression * statement * statement
  | Switch of expression * ((int32 * statement) list) * (statement option)
  | Try_catch of block * ((string * string * block) list)
  | Throw of expression
  | Block of block
  | Synchronized_block of expression * block
and block = statement list

type method_ = {
    meth_javadoc : string list;
    meth_annotations : string list;
    meth_modifiers : modifier list;
    meth_generics : string list;
    meth_return_type : type_ option;
    meth_name : string;
    meth_parameters : (type_ * string) list;
    meth_throws : string list;
    meth_body : block;
  }

val dump_method : ?indent:int -> Buffer.t -> method_ -> unit

type class_ = {
    class_package : string option;
    class_imports : string list;
    class_modifiers : modifier list;
    class_name : string;
    class_extends : string option;
    class_fields : ((modifier list) * type_ * string * (expression option)) list;
    class_static_block : block option;
    class_methods : method_ list;
    class_inner_classes : class_or_enum list;
  }
and enum = {
    enum_modifiers : modifier list;
    enum_name : string;
    enum_values : string list;
  }
and interface = {
    interf_modifiers : modifier list;
    interf_name : string;
    interf_methods : (type_ * string * (type_ list)) list;
  }
and class_or_enum =
  | Full_class of class_
  | Enum of enum
  | Interface of interface

val dump_class : ?indent:int -> Buffer.t -> class_ -> unit

val add_unchecked_class_or_enum : class_or_enum -> class_or_enum

val add_unchecked_class : class_ -> class_

val type_Value : type_

val type_Object : type_

val type_String : type_

val constructor : ?javadoc:string list -> ?annotations:string list -> modifier list -> string -> ?parameters:(type_ * string) list -> block -> method_

val method_ : ?javadoc:string list -> ?annotations:string list -> ?generics:string list -> modifier list -> ?return:type_ option -> string -> ?parameters:(type_ * string) list -> ?throws:string list -> block -> method_

val class_ : modifier list -> string -> ?extends:string option -> ?fields:(modifier list * type_ * string * expression option) list -> ?static:block option -> ?methods:method_ list -> ?inner:class_or_enum list -> unit -> class_

val and_list : expression list -> expression

val return : expression -> statement

val make_equals_body : string -> string -> statement list -> statement list

val make_equals_body_expr_list : string -> string -> expression list -> statement list
