(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

open Ocamlbuild_plugin

let odocl_file = Pathname.pwd / "barista.odocl"
let mlpack_file = Pathname.pwd / "baristaLibrary.mlpack"
let src_path = Pathname.pwd / "src"
let excluded_modules = []

let modules_with_exception = ref []

let () =
  let odocl_chan = open_out odocl_file in
  let mlpack_chan = open_out mlpack_file in
  let add_file filename =
    if (Pathname.check_extension filename "mli")
      || (Pathname.check_extension filename "mly")
      || (Pathname.check_extension filename "mll") then begin
          let module_name = Pathname.remove_extension filename in
          let module_name = Pathname.basename module_name in
          let module_name = String.capitalize module_name in
          if not (List.mem module_name excluded_modules) then begin
            output_string odocl_chan module_name;
            output_char odocl_chan '\n';
            output_string mlpack_chan module_name;
            output_char mlpack_chan '\n'
          end;
          if Pathname.check_extension filename "mli" then begin
            let lines = string_list_of_file (Pathname.mk filename) in
            if List.mem "BARISTA_ERROR" lines then
              modules_with_exception  := module_name :: !modules_with_exception
          end
      end in
  let rec add_files dirname =
    Array.iter
      (fun path ->
        let path = dirname / path in
        if Pathname.is_directory path then
          add_files path
        else
          add_file path)
      (Pathname.readdir dirname) in
  add_files src_path;
  modules_with_exception := List.sort compare !modules_with_exception;
  close_out_noerr odocl_chan;
  close_out_noerr mlpack_chan

let version_tag = "src_common_currentVersion_ml"
let version_ml = "src/common/currentVersion.ml"
let version_file = "../VERSION"
let predefined_ml = "src/driver/predefined.ml"
let commands_path = "../src/commands"

type lib_kind = Byte | Native | Java

let lib_name kind lib =
  lib ^
  (match kind with
  | Byte -> ".cma"
  | Native -> ".cmxa"
  | Java -> ".cmja")

let () =
  let safe_cp src dst =
    let src = Pathname.mk src in
    let dst = Pathname.mk dst in
    let dir = Pathname.dirname dst in
    let cmd = Printf.sprintf "mkdir -p %s" (Pathname.to_string dir) in
    if Sys.command cmd <> 0 then failwith ("cannot run " ^ cmd);
    cp src dst in
  let getenv s def =
    try
      Sys.getenv s
    with _ ->
      def in
  dispatch begin function
    | After_rules ->
        if String.uppercase (getenv "WARNINGS" "FALSE") = "TRUE" then
          flag ["ocaml"; "compile"; "warnings"] (S[A"-w"; A"Ae"; A"-warn-error"; A"A"]);
        flag ["ocaml"; "doc"] (A"-sort");

        let prefix = getenv "PATH_OCAML_PREFIX" "unknown-ocaml-prefix" in

        flag ["ocaml"; "ocamldep"]
          (S[A"-pp"; A(prefix ^ "/bin/camlp4of.opt ../syntax/_build/src/barista_pp.cmxs")]);
        flag ["ocaml"; "compile"; "byte"]
          (S[A"-pp"; A(prefix ^ "/bin/camlp4of.opt ../syntax/_build/src/barista_pp.cmxs")]);
        flag ["ocaml"; "compile"; "native"]
          (S[A"-pp"; A(prefix ^ "/bin/camlp4of.opt ../syntax/_build/src/barista_pp.cmxs")]);
        flag ["ocaml"; "compile"; "java"; "alternative_implementations"]
          (S[A"-pp"; A(prefix ^ "/bin/camlp4of.opt ../syntax/_build/src/barista_pp.cmxs -DUSE_JDK"); A"-java-extensions"; A"-I"; A"+javalib"]);

        flag ["ocaml"; "compile"; "java"; "preprocessor"]
          (S[A"-pp"; A(prefix ^ "/bin/camlp4of.opt ../syntax/_build/src/barista_pp.cmxs")]);

        flag ["ocaml"; "link"; "byte"; "alternative_implementations"]
          (S[A"-I"; A"+camomile"; A(lib_name Byte "camomile")]);
        flag ["ocaml"; "link"; "native"; "alternative_implementations"]
          (S[A"-I"; A"+camomile"; A(lib_name Native "camomile")]);

        flag ["ocaml"; "compile"; "byte"; "alternative_implementations"]
          (S[A"-I"; A"+zip"; A(lib_name Byte "zip")]);
        flag ["ocaml"; "compile"; "native"; "alternative_implementations"]
          (S[A"-I"; A"+zip"; A(lib_name Native "zip")]);

        flag ["ocaml"; "link"; "byte"; "alternative_implementations"]
          (S[A"-I"; A"+zip"; A(lib_name Byte "zip")]);
        flag ["ocaml"; "link"; "native"; "alternative_implementations"]
          (S[A"-I"; A"+zip"; A(lib_name Native "zip")]);


        flag ["ocaml"; "pack"; "java"]
          (S[A"-java-package"; A"fr.x9c.barista"]);

        dep [version_tag] [version_ml];
        rule ("generation of " ^ version_ml)
          ~prod:version_ml
          ~insert:`bottom
          (fun _ _ ->
            let version =
              try
                List.hd (string_list_of_file (Pathname.mk version_file))
              with _ -> "unknown" in
            let name, channel = Filename.open_temp_file "version" ".ml" in
            Printf.fprintf channel "let value = %S\n" version;
            close_out_noerr channel;
            safe_cp name version_ml);

        dep ["src_driver_predefined_ml"] [predefined_ml];
        rule ("generation of " ^ predefined_ml)
          ~prod:predefined_ml
          ~insert:`bottom
          (fun _ _ ->
            let print_module_list channel path =
              let filenames = Sys.readdir path in
              Array.sort Pervasives.compare filenames;
              Array.iter
                (fun filename ->
                  if Filename.check_suffix filename ".mli" then begin
                    let basename = Filename.basename filename in
                    let module_name = String.capitalize (Filename.chop_suffix basename ".mli") in
                    Printf.fprintf channel "  (module %s : Command.T);\n" module_name
                  end)
                filenames in
            let name, channel = Filename.open_temp_file "predefined" ".ml" in
            Printf.fprintf channel "let commands = [\n";
            print_module_list channel commands_path;
            Printf.fprintf channel "]\n\nlet is_barista_exception = function";
            List.iter
              (fun module_name ->
                Printf.fprintf channel "\n  | %s.Exception _" module_name)
              !modules_with_exception;
            Printf.fprintf channel " -> true\n  | _ -> false\n\nlet string_of_exception = function\n";
            List.iter
              (fun module_name ->
                Printf.fprintf channel "  | %s.Exception err -> %s.string_of_error err\n" module_name module_name)
              !modules_with_exception;
            Printf.fprintf channel "  | e -> Printexc.to_string e\n";
            close_out_noerr channel;
            safe_cp name predefined_ml)
    | _ -> ()
  end
