(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Basic implementation of graphs, with output to various formats. *)

(** {6 Graph representation} *)

type vertex = {
    vertex_id : UTF8.t; (** Vertex identifier. *)
    vertex_cluster : UTF8.t option; (** Vertex cluster. *)
    vertex_label : UTF8.t option; (** Vertex label. *)
    vertex_properties : UTF8.t UTF8.Map.t; (** Vertex properties. *)
  }
(** The type of graph vertices. *)
and edge = {
    edge_id : UTF8.t; (** Edge identifier. *)
    edge_directed : bool; (** Whether edge is directed. *)
    edge_vertices : (UTF8.t * UTF8.t) list; (** Vertices (and associated labels) linked by edge (allowing hyperedge). *)
    edge_label : UTF8.t option; (** Edge label. *)
    edge_properties : UTF8.t UTF8.Map.t; (** Edge properties. *)
  }
(** The type of graph edges. *)
and t = {
    vertices : vertex list; (** Graph vertices. *)
    edges : edge list; (** Graph edges. *)
    vertice_types : UTF8.t UTF8.Map.t; (** Association list from property names to their types. *)
    edge_types : UTF8.t UTF8.Map.t; (** Association list from property names to their types. *)
  }
(** The type of graphs. *)


(** {6 Graph builders} *)

type builder
(** The type of graph builder. *)

val make : unit -> builder
(** Creates a builder containing an empty graph. *)

val to_graph : builder -> t
(** Returns the graph as constructed by the passed builder. *)

val add_vertex : ?cluster:UTF8.t ->
  ?label:UTF8.t ->
  ?properties:(UTF8.t * UTF8.t) list ->
  id:UTF8.t ->
  builder ->
  unit
(** Adds a vertex to the passed builder. *)

val add_edge : 
  ?directed:bool ->
  ?label:UTF8.t ->
  ?properties:(UTF8.t * UTF8.t) list ->
  id:UTF8.t ->
  vertices:(UTF8.t * UTF8.t) list ->
  builder ->
  unit
(** Adds an edge to the passed builder. *)

val add_vertice_type : UTF8.t -> UTF8.t -> builder -> unit
(** [add_type n t b] adds the information that properties with name [n]
    have type [t] to builder [b]. *)

val add_edge_type : UTF8.t -> UTF8.t -> builder -> unit
(** [add_type n t b] adds the information that properties with name [n]
    have type [t] to builder [b]. *)


(** {6 Output} *)

type format =
  | Dot (** Dot format ({i http://www.graphviz.org/doc/info/lang.html}). *)
  | GraphML (** GraphML format ({i http://graphml.graphdrawing.org}). *)
  | GEXF (** gexf format ({i http://gexf.net/format/}). *)
(** The type of all output formats. *)

val all_formats : format list
(** The list of all output formats. *)

val string_of_format : format -> string
(** Converts the passed format into a string. *)

val utf8_of_format : format -> UTF8.t
(** Converts the passed format into a UTF8 string. *)

val identifier_of_format : format -> UTF8.t
(** Converts the passed format into a UTF8 string, as a lowercase
    identifier. *)

val dump : format -> t -> UTF8Buffer.t -> unit
(** [dump fmt g buf] dumps the data of graph [g] to buffer [buf], using
    format [fmt]. *)

val map_of_list : (UTF8.t * UTF8.t) list -> UTF8.t UTF8.Map.t
(** Converts an association list into a map. *)
