(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


module type HashableComparableType = sig
  type t
  val equal : t -> t -> bool
  val compare : t -> t -> int
  val hash : t -> int
  val to_string : t -> string
 end

module type S = sig
  type elem
  type t
  type index = Utils.u2
  val make : int -> int -> elem -> t
  val from_array : exn -> elem array -> elem -> t
  val to_array : t -> elem array
  val length : t -> int
  val capacity : t -> int
  val get : t -> Utils.u2 -> elem
  val set : t -> Utils.u2 -> elem -> unit
  val find : elem -> t -> Utils.u2
  val add : exn -> t -> elem -> elem -> bool -> Utils.u2
  val add_if_not_found : exn -> t -> elem -> elem -> bool -> Utils.u2
  val equal : t -> t -> bool
  val compare : t -> t -> int
  val hash : t -> int
  val to_string : t -> string
end

external unsafe_u2 : int -> Utils.u2 =
  "%identity"

module Make (T : HashableComparableType) : (S with type elem = T.t) = struct

  module ElementTable = Hashtbl.Make (T)

  type elem = T.t

  type t = {
      ext_array : elem ExtendableArray.t;
      inv_array : Utils.u2 ElementTable.t
    }

  type index = Utils.u2

  let make len cap init =
    { ext_array = ExtendableArray.make len cap init;
      inv_array = ElementTable.create 997 }

  let from_array exn arr x =
    let ext_array = ExtendableArray.from_array exn arr x in
    let inv_array = ElementTable.create 997 in
    for idx = 0 to pred (Array.length arr) do
      ElementTable.add inv_array arr.(idx) (unsafe_u2 idx)
    done;
    { ext_array; inv_array }

  let to_array arr =
    ExtendableArray.to_array arr.ext_array

  let length arr =
    ExtendableArray.length arr.ext_array

  let capacity arr =
    ExtendableArray.capacity arr.ext_array

  let get arr idx =
    ExtendableArray.get arr.ext_array idx

  let set arr idx x =
    ExtendableArray.set arr.ext_array idx x;
    ElementTable.add arr.inv_array x idx

  let find x arr =
    ElementTable.find arr.inv_array x

  let add exn arr x z addit =
    let idx = ExtendableArray.add exn arr.ext_array x z addit in
    ElementTable.add arr.inv_array x idx;
    idx

  let add_if_not_found exn arr x z addit =
    try
      find x arr
    with Not_found ->
      add exn arr x z addit

  let equal arr1 arr2 =
    ExtendableArray.equal T.equal arr1.ext_array arr2.ext_array

  let compare arr1 arr2 =
    ExtendableArray.compare T.compare arr1.ext_array arr2.ext_array

  let hash arr =
    ExtendableArray.hash T.hash arr.ext_array

  let to_string arr =
    ExtendableArray.to_string T.to_string arr.ext_array

end
