(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Utility functions and modules used throughout the Barista project.

    This module defines a bunch of [u]{i n} and [s]{i n} types that alias
    integer types. They are defined to keep the type expressions as close
    as possible to the Java specification. Moreover, these types are
    declared private and should be constructed by the functions with the
    same sames. *)


(** {6 Operators} *)

val (+=) : int ref -> int -> unit
(** [x += k] is equivalent to [x := !x + k]. *)

val (-=) : int ref -> int -> unit
(** [x -= k] is equivalent to [x := !x - k]. *)


(** {6 Integer types} *)

(** {7 Integer erros} *)

type integer_error = {
    type_name : string; (** Integer type name. *)
    lower_bound : int64; (** Lower bound of integer type. *)
    upper_bound : int64; (** Upper bound of integer type. *)
    value : int64; (** Actual value passed to constructor function. *)
  }
(** Error describing an integer exception. *)

exception Integer_exception of integer_error
(** Raised when an attempt made to construct an integer out of bounds. *)

val string_of_integer_error : integer_error -> string
(** Converts the passed error into a string. *)

(** {7 OCaml integers} *)

val min_int_value : int
(** The lowest [int] value. *)

val max_int_value : int
(** The greatest [int] value. *)

val min_int : int -> int -> int
(** Minimum function over [int] values. *)

val max_int : int -> int -> int
(** Maximum function over [int] values. *)

(** {7 u1} *)

type u1 = private int
(** The type of unsigned 1-byte integers. *)

val u1 : int -> u1
(** Constructs a value of type [u1], raising [Integer_exception] if value
    is out of bounds. *)

val min_u1_value : int
(** The lowest [u1] value. *)

val max_u1_value : int
(** The greatest [u1] value. *)

val min_u1 : u1 -> u1 -> u1
(** Minimum function over [u1] values. *)

val max_u1 : u1 -> u1 -> u1
(** Maximum function over [u1] values. *)

val u1_succ : u1 -> u1
(** Increments the passed value, raising [Integer_exception] if result
    would be out of bounds. *)

val u1_pred : u1 -> u1
(** Decrements the passed value, raising [Integer_exception] if result
    would be out of bounds. *)

(** {7 s1} *)

type s1 = private int
(** The type of signed 1-byte integers. *)

val s1 : int -> s1
(** Constructs a value of type [s1], raising [Integer_exception] if value
    is out of bounds. *)

val min_s1_value : int
(** The lowest [s1] value. *)

val max_s1_value : int
(** The greatest [s1] value. *)

val min_s1 : s1 -> s1 -> s1
(** Minimum function over [u1] values. *)

val max_s1 : s1 -> s1 -> s1
(** Maximum function over [u1] values. *)

val s1_succ : s1 -> s1
(** Increments the passed value, raising [Integer_exception] if result
    would be out of bounds. *)

val s1_pred : s1 -> s1
(** Decrements the passed value, raising [Integer_exception] if result
    would be out of bounds. *)

val s1_neg : s1 -> s1
(** Unary negation,
    raises [Integer_exception] if result would be out of bounds. *)

(** {7 u2} *)

type u2 = private int
(** The type of unsigned 2-byte integers. *)

val u2 : int -> u2
(** Constructs a value of type [u2], raising [Integer_exception] if value
    is out of bounds. *)

val min_u2_value : int
(** The lowest [u2] value. *)

val max_u2_value : int
(** The greatest [u2] value. *)

val min_u2 : u2 -> u2 -> u2
(** Minimum function over [u2] values. *)

val max_u2 : u2 -> u2 -> u2
(** Maximum function over [u2] values. *)

val u2_succ : u2 -> u2
(** Increments the passed value, raising [Integer_exception] if result
    would be out of bounds. *)

val u2_pred : u2 -> u2
(** Decrements the passed value, raising [Integer_exception] if result
    would be out of bounds. *)

(** {7 s2} *)

type s2 = private int
(** The type of signed 2-byte integers. *)

val s2 : int -> s2
(** Constructs a value of type [s2], raising [Integer_exception] if value
    is out of bounds. *)

val min_s2_value : int
(** The lowest [s2] value. *)

val max_s2_value : int
(** The greatest [s2] value. *)

val min_s2 : s2 -> s2 -> s2
(** Minimum function over [s2] values. *)

val max_s2 : s2 -> s2 -> s2
(** Maximum function over [s2] values. *)

val s2_succ : s2 -> s2
(** Increments the passed value, raising [Integer_exception] if result
    would be out of bounds. *)

val s2_pred : s2 -> s2
(** Decrements the passed value, raising [Integer_exception] if result
    would be out of bounds. *)

val s2_neg : s2 -> s2
(** Unary negation,
    raises [Integer_exception] if result would be out of bounds. *)

(** {7 u4} *)

type u4 = private int64
(** The type of unsigned 4-byte integers. *)

val u4 : int64 -> u4
(** Constructs a value of type [u4], raising [Integer_exception] if value
    is out of bounds. *)

val min_u4_value : int64
(** The lowest [u4] value. *)

val max_u4_value : int64
(** The greatest [u4] value. *)

val min_u4 : u4 -> u4 -> u4
(** Minimum function over [u4] values. *)

val max_u4 : u4 -> u4 -> u4
(** Maximum function over [u4] values. *)

val u4_succ : u4 -> u4
(** Increments the passed value, raising [Integer_exception] if result
    would be out of bounds. *)

val u4_pred : u4 -> u4
(** Decrements the passed value, raising [Integer_exception] if result
    would be out of bounds. *)

(** {7 s4} *)

type s4 = private int32
(** The type of signed 4-byte integers. *)

external s4 : int32 -> s4 = "%identity"
(** Identity function. *)

val min_s4_value : s4
(** The lowest [s4] value. *)

val max_s4_value : s4
(** The greatest [s4] value. *)

val min_s4 : s4 -> s4 -> s4
(** Minimum function over [s4] values. *)

val max_s4 : s4 -> s4 -> s4
(** Maximum function over [s4] values. *)

val s4_succ : s4 -> s4
(** Increments the passed value, raising [Integer_exception] if result
    would be out of bounds. *)

val s4_pred : s4 -> s4
(** Decrements the passed value, raising [Integer_exception] if result
    would be out of bounds. *)

val s4_neg : s4 -> s4
(** Unary negation,
    raises [Integer_exception] if result would be out of bounds. *)

(** {7 s8} *)

type s8 = private int64
(** The type of signed 8-byte integers. *)

external s8 : int64 -> s8 = "%identity"
(** Identity function. *)

val min_s8_value : int64
(** The lowest [s8] value. *)

val max_s8_value : int64
(** The greatest [s8] value. *)

val min_s8 : s8 -> s8 -> s8
(** Minimum function over [s8] values. *)

val max_s8 : s8 -> s8 -> s8
(** Maximum function over [s8] values. *)

val s8_succ : s8 -> s8
(** Increments the passed value, raising [Integer_exception] if result
    would be out of bounds. *)

val s8_pred : s8 -> s8
(** Decrements the passed value, raising [Integer_exception] if result
    would be out of bounds. *)

val s8_neg : s8 -> s8
(** Unary negation,
    raises [Integer_exception] if result would be out of bounds. *)

val split_s8 : s8 -> (s4 * s4)
(** Splits the passed 8-byte value into two 4-byte values,
    the first containing the highest bits. *)

val gather_s8 : s4 -> s4 -> s8
(** Gather two 4-byte values into a 8-byte value,
    the first 4-byte value containing the highest bits. *)

(** {7 Conversions} *)

external u2_of_u1 : u1 -> u2 = "%identity"
(** Identity function. *)

external s4_of_s2 : s2 -> s4 = "%int32_of_int"
(** Identity function. *)


(** {6 List and array utilities} *)

val list_mem : ('a -> 'a -> bool) -> 'a -> 'a list -> bool
(** Similar to [List.mem] except that the first parameter is the
    equality function over element. *)

val list_assoc : ('a -> 'a -> bool) -> 'a -> ('a * 'b) list -> 'b
(** Similar to [List.assoc] except that the first parameter is the
    equality function over keys. *)

val map_partial : ('a -> 'b option) -> 'a list -> 'b list
(** Similar to [List.map] except that the returned list contains the
    elements mapped to [Some x], and ignores those mapped to [None].

    {i Not tail-recursive.} *)

val map_list_to_array : ('a -> 'b) -> 'a list -> 'b array
(** [map_list_to_array f l] is equivalent to [Array.map f (Array.of_list l)]. *)

val map_array_to_list : ('a -> 'b) -> 'a array -> 'b list
(** [map_array_to_list f a] is equivalent to [Array.to_list (List.map f l)]. *)

val list_equal : ('a -> 'a -> bool) -> 'a list -> 'a list -> bool
(** Equality over lists, the first parameter being the predicate used to
    compare list elements. *)

val list_compare : ('a -> 'a -> int) -> 'a list -> 'a list -> int
(** Comparison over lists, the first parameter being the comparison
    function over list elements. *)

val list_hash : ('a -> int) -> 'a list -> int
(** Hash over lists, the first parameter being the hash function over
    list elements. *)

val array_equal : ('a -> 'a -> bool) -> 'a array -> 'a array -> bool
(** Equality over arrays, the first parameter being the predicate used to
    compare array elements. *)

val array_compare : ('a -> 'a -> int) -> 'a array -> 'a array -> int
(** Comparison over arrays, the first parameter being the comparison
    function over array elements. *)

val array_hash : ('a -> int) -> 'a array -> int
(** Hash over arrays, the first parameter being the hash function over
    array elements. *)


(** {6 String utilities} *)

val string_replace : char -> char -> string -> string
(** [string_replace o n s] returns a new string where every occurrence of [o]
    is replaced by [n]. *)

val string_split : string -> string -> string list
(** [string_split seps s] returns a list of substrings of [s] separeted by
    contiguous substrings of elements from [seps]. *)

val string_of_list : ('a -> string) -> 'a list -> string
(** [string_of_list f l] converts [l] into a string, using [f] to convert
    list elements. *)

val string_of_array : ('a -> string) -> 'a array -> string
(** [string_of_array f a] converts [a] into a string, using [f] to
    convert array elements. *)

val hash_string : string -> int
(** Hash function over strings. *)


(** {6 Miscellaneous} *)

val fix_point : ('a -> 'a -> bool) -> ('a -> 'a) -> 'a -> 'a
(** [fix_point eq f x] returns a fix point of [f], seeding with value [x]
    and using [eq] as the equality function. *)

val compose_list : ('a -> 'a) list -> 'a -> 'a
(** [compose_list [f1; f2; ...; fn] x] returns [f1 (f2 ... (fn x))]. *)

val try_finally : 'a -> ('a -> 'b) -> ('a -> unit) -> 'b
(** [try_finally x f h] implements the try/finally logic.
    [f] is the body of the try clause, while [h] is the finally handler.

    Both [f] and [h] receive [x] as their parameter. *)

val identity : 'a -> 'a
(** The identity function. *)

val switch : ('a -> 'a -> bool) -> ('a * ('a -> 'b)) list -> ('a -> 'b) -> 'a -> 'b
(** [switch eq matches default x] search the association list [matches]
    for a key equal to [x] using equality function [eq]. If such a key
    exists, the return value is equal to the application of the
    associated function to [x]. Otherwise, the return value is equal to
    the application of [default] to [x]. *)

val universal_hash : 'a -> int
(** Universal hash function, bare shorthand for [Hashtbl.hash]. *)

val make_header_printer : ('a -> string) -> 'a list -> ('a -> unit)
(** [make_header_printer conv l] returns a function to print headers on
    the standard output. Nothing will be printed by the returned function
    if [List.length l <= 1], otherwise the passed string will be printed
    enclosed with dashes. [conv] is used to convert list elements into
    strings. *)
