(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Class loaders that allows easy access and loading of classes,
    packages, and modules. *)


(** {6 Types} *)

type t
(** The type of class loaders, that are barely maps from names (in
    external form) to their definitions, in each category (class,
    package, and module). *)

BARISTA_ERROR =
  | Unable_to_load of UTF8.t * string
  | Already_defined of UTF8.t


(** {6 Constructor} *)

val make_of_class_path : ClassPath.t -> t
(** Constructs a class loader from a class path that will be used to load
    elements. The returned class loader contains no definition. *)


(** {6 Lookup} *)

val find_class : t -> UTF8.t -> ClassDefinition.t
(** [find_class loader cn] returns the class definition for the class
    named [cn] (in external form) using class loader [loader].

    Raises [Exception] if such a class has not already been loaded and
    cannot be loaded. *)

val find_class_name : t -> Name.for_class -> ClassDefinition.t
(** [find_class_name loader cn] returns the class definition for the
    class named [cn] (in external form) using class loader [loader].

    Raises [Exception] if such a class has not already been loaded and
    cannot be loaded. *)

val find_package : t -> UTF8.t -> PackageDefinition.t
(** [find_package loader pn] returns the package definition for the
    package named [pn] (in external form) using class loader [loader].

    Raises [Exception] if such a package has not already been loaded and
    cannot be loaded. *)

val find_package_name : t -> Name.for_package -> PackageDefinition.t
(** [find_package_name loader pn] returns the package definition for the
    package named [pn] (in external form) using class loader [loader].

    Raises [Exception] if such a package has not already been loaded and
    cannot be loaded. *)

val find_module : t -> UTF8.t -> ModuleDefinition.t
(** [find_module loader pn] returns the module definition for the module
    named [mn] (in external form) using class loader [loader].

    Raises [Exception] if such a module has not already been loaded and
    cannot be loaded. *)
val find_module_name : t -> Name.for_module -> ModuleDefinition.t
(** [find_module_name loader pn] returns the module definition for the
    module named [mn] (in external form) using class loader [loader].

    Raises [Exception] if such a module has not already been loaded and
    cannot be loaded. *)


(** {6 Addition} *)

val add_class : t -> ClassDefinition.t -> unit
(** [add_class loader cd] adds the class [cd] to the class loader
    [loader].

    Raises [Exception] if a class with the same name has already been
    loaded. *)

val add_package : t -> PackageDefinition.t -> unit
(** [add_package loader pd] adds the package [pd] to the class loader
    [loader].

    Raises [Exception] if a package with the same name has already been
    loaded. *)

val add_module : t -> ModuleDefinition.t -> unit
(** [add_module loader md] adds the module [md] to the class loader
    [loader].

    Raises [Exception] if a module with the same name has already been
    loaded. *)


(** {6 Test} *)

val mem_class : t -> UTF8.t -> bool
(** [mem_class loader cn] tests whether the class named [cn] has been
    loaded by the class loader [loader]. *)

val mem_class_name : t -> Name.for_class -> bool
(** [mem_class_name loader cn] tests whether the class named [cn] has
    been loaded by the class loader [loader]. *)

val mem_package : t -> UTF8.t -> bool
(** [mem_package loader pn] tests whether the package named [pn] has been
    loaded by the class loader [loader]. *)

val mem_package_name : t -> Name.for_package -> bool
(** [mem_package_name loader pn] tests whether the package named [pn] has
    been loaded by the class loader [loader]. *)

val mem_module : t -> UTF8.t -> bool
(** [mem_module cl mn] tests whether the module named [mn] has been
    loaded by the class loader [loader]. *)

val mem_module_name : t -> Name.for_module -> bool
(** [mem_module_name cl mn] tests whether the module named [mn] has been
    loaded by the class loader [loader]. *)


(** {6 Removal} *)

val remove_class : t -> UTF8.t -> unit
(** [remove_class loader cn] removes the definition for the class named
    [cn] from class loader [loader]. *)

val remove_class_name : t -> Name.for_class -> unit
(** [remove_class_name loader cn] removes the definition for the class
    named [cn] from class loader [loader]. *)

val remove_package : t -> UTF8.t -> unit
(** [remove_package loader pn] removes the definition for the package
    named [pn] from class loader [loader]. *)

val remove_package_name : t -> Name.for_package -> unit
(** [remove_package_name loader pn] removes the definition for the
    package named [pn] from class loader [loader]. *)

val remove_module : t -> UTF8.t -> unit
(** [remove_module loader mn] removes the definition for the module named
    [mn] from class loader [loader]. *)

val remove_module_name : t -> Name.for_module -> unit
(** [remove_module_name loader mn] removes the definition for the module
    named [mn] from class loader [loader]. *)


(** {6 Replacement} *)

val replace_class : t -> ClassDefinition.t -> unit
(** Equivalent to [add_class] except that a previous definition is
    replaced. *)

val replace_package : t -> PackageDefinition.t -> unit
(** Equivalent to [add_package] except that a previous definition is
    replaced. *)

val replace_module : t -> ModuleDefinition.t -> unit
(** Equivalent to [add_module] except that a previous definition is
    replaced. *)
