(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

IFNDEF USE_JDK THEN

let (|>) x f = f x

external string_of_bytes : Bytes.t -> string =
  "%identity"

external bytes_of_string : string -> Bytes.t =
  "%identity"

type t = Buffer.t

let make_of_size sz =
  if sz < 0 then
    invalid_arg "BaristaLibrary.ByteBuffer.make_of_size";
  Buffer.create sz

let add_byte buff byte =
  byte land 0x000000FF
  |> char_of_int
  |> Buffer.add_char buff

let add_bytes buff bytes =
  bytes
  |> string_of_bytes
  |> Buffer.add_string buff

let add_bytes_from buff bytes ofs len =
  let bytes = string_of_bytes bytes in
  if (ofs < 0) || (len < 0) || ((ofs + len) > (String.length bytes)) then
    invalid_arg "ByteBuffer.add_bytes_from";
  Buffer.add_substring
    buff
    bytes
    ofs
    len

let contents buff =
  buff
  |> Buffer.contents
  |> bytes_of_string

ELSE (* USE_JDK *)

let (|>) x f = f x

type t = java'io'ByteArrayOutputStream java_instance

external bytes_of_byte_array : int JavaByteArray.t -> Bytes.t =
  "%identity"

external byte_array_of_bytes : Bytes.t -> int JavaByteArray.t =
  "%identity"

let make_of_size sz =
  if sz < 0 then
    invalid_arg "BaristaLibrary.ByteBuffer.make_of_size";
  sz
  |> Int32.of_int
  |> Java.make "java.io.ByteArrayOutputStream(_)"

let add_byte buff byte =
  byte
  |> Int32.of_int
  |> Java.call "java.io.ByteArrayOutputStream.write(int)" buff

let add_bytes buff bytes =
  bytes
  |> byte_array_of_bytes
  |> Java.call "java.io.ByteArrayOutputStream.write(byte[])" buff

let add_bytes_from buff bytes ofs len =
  let bytes = byte_array_of_bytes bytes in
  let len_bytes = JavaByteArray.length bytes in
  if (ofs < 0) || (len < 0) || ((ofs + len) > (Int32.to_int len_bytes)) then
    invalid_arg "BaristaLibrary.ByteBuffer.add_bytes_from";
  Java.call "java.io.ByteArrayOutputStream.write(byte[],_,_)"
    buff
    bytes
    (Int32.of_int ofs)
    (Int32.of_int len)

let contents buff =
  buff
  |> Java.call "java.io.ByteArrayOutputStream.toByteArray()"
  |> bytes_of_byte_array

END
